import moment from 'moment';

import { FormatDateInString } from '../../../ui/FormatDate';
import { WebphoneLogsDB } from './webphoneLogsDB';

const freshLogs = [1, 'hour'];
const logObjectSpace = 2;
type TWebLogLine = {content: string[]; date: number};

export default class WebphoneLogs {
    private webphoneDB;
    hasCall;

    constructor() {
        this.webphoneDB = new WebphoneLogsDB();
    }

    /*
        From here logLine is a string array, where 1st element contains a message,
        all other have formatting arguments
     */
    async writeLog(logLine: string[]) {
        await this._writeWebphoneLog(logLine);
    }

    async getLogsForDownload(): Promise<string[]> {
        return WebphoneLogs._formatLogText(await this._getLog());
    }

    _getLog(): Promise<TWebLogLine[]> {
        return this.webphoneDB.mainLogs
            ?.toArray()
            ?.then((response) => {
                response = response.map((el) => {
                    try {
                        return { date: el.date, content: JSON.parse(el.content) };
                    } catch(e) {
                        return { date: el.date, content: [`Error: ${e}`] };
                    }
                });

                return response;
            }) ?? [];
    }

    /* not private for testing */
    static _formatLogText(logs: TWebLogLine[]) {
        const textLogs: string[] = [];
        logs.forEach((logLine: TWebLogLine) => {
            const logDate = FormatDateInString({ value: logLine.date, withSecond: true });
            if (logLine.content?.length > 0) {
                let result = logLine.content?.[0]?.replace(/([><])/g, '');
                let formattingSpecifiersCount = 0;
                const formattingSpecifiersRegex = /(%[a-z,%])((\w|:)+)?/g;
                const formatter = (formattingArg, match, formatSpecifier, content) => {
                    switch (formatSpecifier) {
                    case '%c':
                        return content ?? '';
                    case '%o':
                        let parsedObject: string;
                        try {
                            parsedObject = JSON.stringify(formattingArg, null, logObjectSpace);
                        } catch (e) {
                            parsedObject = '';
                        }

                        return parsedObject;
                    case '%s':
                        return formattingArg ?? '';
                    case '%%':
                        return `%${content ?? ''}`;
                    default:
                        return match;
                    }
                };

                result = result.replace(formattingSpecifiersRegex, (match, formatSpecifier, content) => {
                    formattingSpecifiersCount++;

                    return formatter(logLine.content?.[formattingSpecifiersCount], match, formatSpecifier, content);
                });

                /* formatting line to separate log messages */
                textLogs.push(`${logDate}\n${result}\n------------\n`);
            }
        });

        return textLogs;
    }

    private async _writeWebphoneLog(logLine: string[]) {
        this.clearOutdatedWebphoneLog();
        let mainLogsLine = '[]';

        try {
            mainLogsLine = JSON.stringify(logLine);
        } catch(e) {
            throw `Error: ${e}`;
        }

        const logsLineToDB = {
            date: Date.now(),
            content: mainLogsLine,
        };

        await this.webphoneDB.mainLogs.put(logsLineToDB);
    }

    private clearOutdatedWebphoneLog() {
        const limit = +moment().subtract(...freshLogs);

        return this.webphoneDB.mainLogs
            ?.where('date')
            ?.between(0, limit)
            ?.delete();
    }
}
