import { Howl } from 'howler';
import { C, debug, UA, WebSocketInterface } from 'jssip';
import { IceCandidateEvent } from 'jssip/lib/RTCSession';
import moment from 'moment';
import { batch } from 'react-redux';

import {
    endAction,
    IWebphoneReducerState,
    setCallStatus,
    setCustomCallDescription,
    setWebphoneStatus,
} from '../../../reducers/webphoneReducer';
import { Request2 } from '../../../utils/request';
import { ytLogs } from '../../../utils/sendLogs';
import { EVENT_TYPES } from '../../../utils/sendLogs/eventTypes/eventTypes';
import { ErrorSource, logError } from '../../Content/initErrorCounter';
import WebphoneLogs from '../helpers/webphoneLogs';
import { REQUESTS, WEBPHONE_REQUESTS as requestConfigs } from '../request';
import { CALL_STATUS, callStatusDescriptions, IGetCallIdProps, WEBPHONE_STATUS } from './webphoneTypes';

const SERVER = 'drive-phone.tel.yandex-team.ru';
const stunConfig = [{ 'urls': ['stun:141.8.146.81:3478'] }];
const PORT = 7443;

interface IWebphoneMainProps {
    dispatch: any;
    webphoneStore: IWebphoneReducerState;
    yandexLogin: string;
    setError: () => void;
}

debug.enable('JsSIP:*');
const logger = new WebphoneLogs();
debug.log = async (...args) => {
    await logger.writeLog.call(logger, args);
};

export class WebphoneMain {
    private socket;
    private _ua;
    private configuration;
    private readonly dispatch;
    private session;
    private webphoneData: IWebphoneReducerState;
    private readonly yandexLogin;
    private readonly setError;
    private request = new Request2({ requestConfigs });
    private log = ytLogs.getInstance();

    constructor(props: IWebphoneMainProps) {
        this.dispatch = props.dispatch;
        this.webphoneData = props.webphoneStore;
        this.yandexLogin = props.yandexLogin;
        this.setError = props.setError;
    }

    private peerConnectionHandler({ peerconnection }: { peerconnection: RTCPeerConnection }) {
        peerconnection.addEventListener('track', (event: RTCTrackEvent) => {
            const audio = new Audio(); // создадим новое аудио
            const [stream] = event.streams;

            audio.srcObject = stream; // прокидываем в него стрим
            audio.play();
        });
    }

    private checkIfMicrophoneIsAllowed(): Promise<boolean> {
        return navigator.mediaDevices.getUserMedia({ audio: true, video: false })
            .then(() => true)
            .catch((error) => {
                this.setError(error);

                return false;
            });
    }

    public async getCallId(props: IGetCallIdProps): Promise<{callId; callTagId} | undefined> {
        const { callRecipient, chatTagId } = props;

        try {
            const response = await this.request.exec(REQUESTS.GET_CALL_ID, {
                queryParams: {
                    ...callRecipient,
                    tag_id: chatTagId?.length ? chatTagId : null,
                },
            });

            const callId = response.call_id;
            const callTagId = response.tag_id;

            return { callId, callTagId };
        } catch(e) {
            this.setError(e);

            return;
        }
    }

    private async dropCallPerformer(tagId) {
        try {
            await this.request.exec(REQUESTS.DROP_PERFORM, {
                body: {
                    drop_tag_ids: tagId,
                },
            });
        }
        catch {}
    }

    /* Additional check for error logging.
        When incoming calls are implemented, this check should be performed on a timer  */
    private static checkStunServer(stunConfig: RTCIceServer[]): Promise<{ ok: boolean; error? }> {
        return new Promise((resolve, reject) => {
            const PeerConnection = window.RTCPeerConnection || window.webkitRTCPeerConnection;
            try {
                const pc = new PeerConnection({ iceServers: stunConfig });
                pc.createDataChannel('');
                pc.createOffer().then(sdp => {
                    pc.setLocalDescription(sdp);
                });
                pc.addEventListener('icecandidate', ice => {
                    if (!ice?.candidate?.candidate) {
                        return;
                    }

                    if (ice.candidate.candidate.includes('typ')) {
                        resolve({ ok: true });
                        pc.close();
                    }
                });
            } catch (error) {
                reject({ ok: false, error });
            }
        });
    }

    public endCall() {
        this.session?.terminate();
    }

    public register() {
        this.socket = new WebSocketInterface(`wss://${SERVER}:${PORT}`);
        this.configuration = {
            sockets: this.socket,
            session_timers: false,
            authorization_user: this.yandexLogin,
            uri: `sip:${this.yandexLogin}@yandex-team.ru`,
            password: `NA`,
        };
        this._ua = new UA(this.configuration);
        this._ua.start();

        this._ua.on('newRTCSession', (data) => {
            this.session = data.session;
            this.session.on('peerconnection', this.peerConnectionHandler);
            /*
                DRIVEFRONT-1061
                Don't wait on all candidates, allows to connect quicker when using VPN
                https://jssip.net/documentation/3.2.x/api/session/#event_icecandidate
                https://stackoverflow.com/questions/61253322/a-40-sec-delay-of-sip-call-initiation-using-jssip-webrtc
            */
            this.session.on('icecandidate', async (event: IceCandidateEvent) => {
                event.ready();
                await logger
                    .writeLog([`WebphoneMain:newRTCSession:icecandidate (custom):  ${event.candidate.candidate}`]);
            });
        });

        this._ua.on('connecting', () => {
            this.dispatch(setWebphoneStatus(WEBPHONE_STATUS.connecting));
        });

        this._ua.on('connected', () => {
            this.dispatch(setWebphoneStatus(WEBPHONE_STATUS.connected));
        });

        this._ua.on('registered', () => {
            this.dispatch(setWebphoneStatus(WEBPHONE_STATUS.registered));
        });

        this._ua.on('registrationFailed', (data) => {
            logError(data.cause, ErrorSource.SIP_ERROR_REGISTRATION, {
                data,
                hasActiveCall: this.session?.isEstablished() || this.session?.isInProgress(),
                username: data?.response?.headers?.To?.[0]?.parsed?._uri?._user,
            });

            if (this.session?.isEstablished() || this.session?.isInProgress()) {
                this.dispatch(setWebphoneStatus(WEBPHONE_STATUS.registrationFailedDuringCall));
            } else {
                this.dispatch(setWebphoneStatus(WEBPHONE_STATUS.registrationFailed));
                this.setError(new Error(data.cause));
            }
        });
    }

    private static playEndCallSound() {
        const sound = new Howl({
            src: ['https://carsharing.s3.yandex.net/admin/static/webphone/call-ended.mp3'],
            volume: 0.8,
        });
        sound.play();
    }

    private async updateCallStatus(callId, backendStatus, comment?) {
        const body = {
            call_id: callId,
            call_status: backendStatus,
        };

        if (comment) {
            body['comment'] = comment;
        }

        try {
            await this.request.exec(REQUESTS.CALL_UPDATE, {
                body,
            });
        }
        catch {}
    }

    private static async logStatus(statusLine) {
        await logger.writeLog([`\n\n\t\t${statusLine} - ${moment().format('YYYY-MM-DD / HH:mm:ss')}\n\n`]);
    }

    public async call(
        props: { client: string; callInfo: IGetCallIdProps },
        callId: string,
        callTagId: string,
        mobIdHeader?: string,
    ) {
        const { client } = props;
        await WebphoneMain.logStatus('Начало звонка');
        if (await this.checkIfMicrophoneIsAllowed()) {
            const extraHeaders = [`X-Call-ID: ${callId}`];

            if (mobIdHeader) {
                extraHeaders.push(mobIdHeader);
            }

            if (callId) {
                this.dispatch(setCustomCallDescription(null));
                const stunServerState = await WebphoneMain.checkStunServer(stunConfig);

                if (stunServerState?.ok) {
                    this.session = this._ua.call(client, {
                        extraHeaders,
                        pcConfig:
                            {
                                hackStripTcp: true, // Какая-то проблема с хром
                                rtcpMuxPolicy: 'negotiate', // multiplexing
                                iceServers: stunConfig,
                            },
                        rtcOfferConstraints:
                            {
                                offerToReceiveAudio: 1, // Принимаем только аудио
                                offerToReceiveVideo: 0,
                            },
                        mediaConstraints: {
                            audio: true,
                            video: false,
                        },
                        eventHandlers: {
                            peerconnection: this.peerConnectionHandler,
                        },
                    });

                    this.session.on('connecting', async () => {
                        this.dispatch(setCallStatus(CALL_STATUS.connecting));
                    });

                    this.session.on('progress', async () => {
                        this.dispatch(setCallStatus(CALL_STATUS.progress));
                        await WebphoneMain.logStatus('Дозвон');
                        this.log.send({
                            data: {
                                event_type: EVENT_TYPES.WEBPHONE_INVITE_SENT,
                                call_id: callId,
                            },
                        });
                    });

                    this.session.on('failed', async (event) => {
                        if (event?.cause === C.causes.BUSY) {
                            this.dispatch(setCustomCallDescription(callStatusDescriptions.busy));
                            await WebphoneMain.logStatus(`Абонент занят - ${event.cause}`);
                        } else {
                            await WebphoneMain.logStatus(`Завершен - ${event.cause}`);
                        }

                        this.dispatch(setCallStatus(CALL_STATUS.failed));
                        await this.dropCallPerformer(callTagId);
                        await this.updateCallStatus(callId, 'not_servised', event);

                        if (event?.originator != 'local') {
                            WebphoneMain.playEndCallSound();
                        }

                        if (event?.cause !== C.causes.BUSY
                            && event?.cause !== C.causes.REJECTED
                            && event?.cause !== C.causes.CANCELED
                            && event?.cause !== C.causes.UNAVAILABLE
                        ) {
                            logError(new Error(event.cause), ErrorSource.SIP_ERROR, {
                                ...event, callId,
                            });
                        }
                    });

                    this.session.on('ended', async (event) => {
                        if (event?.originator === 'remote' && event?.cause === C.causes.BYE) {
                            this.dispatch(setCustomCallDescription(callStatusDescriptions.endedByUser));
                            await WebphoneMain.logStatus(`Звонок завершен клиентом - ${event.cause}`);
                        } else {
                            await WebphoneMain.logStatus(`Завершен - ${event.cause}`);
                        }

                        this.dispatch(setCallStatus(CALL_STATUS.ended));
                        await this.dropCallPerformer(callTagId);
                        await this.updateCallStatus(callId, 'completed');

                        if (event?.originator != 'local') {
                            WebphoneMain.playEndCallSound();
                        }
                    });

                    this.session.on('accepted', () => {
                        this.dispatch(setCallStatus(CALL_STATUS.accepted));
                    });
                } else {
                    this.setError(stunServerState.error);
                    logError(stunServerState.error, ErrorSource.SIP_ERROR, { callId });
                    await logger.writeLog([`WebphoneMain:ERROR:checkStunServer (custom):  ${stunServerState.error}`]);
                }
            }
        }
    }

    public holdCall(isHoldActive: boolean) {
        if (isHoldActive) {
            this.session.unhold();
        } else {
            this.session.hold();
        }
    }

    public muteCall(isMuteActive: boolean) {
        if (isMuteActive) {
            this.session.unmute();
        } else {
            this.session.mute();
        }
    }

    public closeWebphone() {
        this._ua?.stop?.();
        this._ua?.unregister?.();
        batch(() => {
            this.dispatch(endAction());
            this.dispatch(setWebphoneStatus(WEBPHONE_STATUS.notRegistered));
            this.dispatch(setCallStatus(null));
        });
    }
}
