import * as React from 'react';

import { Dict } from '../../types';
import { IProgressBarProps } from '../components/ProgressBar/types';

export const requestQueueWithAllSettled = async (
    queue: () => Promise<unknown>[],
    errorHandler: (error: Error) => void,
) => {
    let response;

    if (Promise.allSettled) {
        const _response = await Promise.allSettled(queue());
        response = _response.map((response) => {
            if (response.status === 'fulfilled') {
                return response.value;
            }

            errorHandler(response.reason);

            return {};

        });
    } else {
        try {
            response = await Promise.all(queue());
        } catch (error) {
            errorHandler(error);
        }
    }

    return response;
};

export interface IRequestQueueItem {
    requestName: string;
    requestOptions: Dict<unknown>;
}

export function useRequestQueueHandler<T>(
    request,
    requestQueueOptions: IRequestQueueItem[],
    responseHandler?,
): [boolean, T | null, Error[], () => void, IProgressBarProps] {
    const [isLoading, setLoading] = React.useState(false);
    const [response, setResponse] = React.useState<T | null>(null);
    const [errors, setErrors] = React.useState<Error[]>([]);
    const [requestShouldFire, fireRequest] = React.useState<boolean>(false);
    const [progressBarObject, setProgressBarObject] = React.useState<IProgressBarProps>({
        allLength: 0,
        successLength: 0,
        errors: [] as { data: number; error: Error }[],
    });

    const getQueue = () => {
        setProgressBarObject(prev => {
            return {
                ...prev,
                successLength: 0,
                errors: [],
            };
        });

        return requestQueueOptions.map(({ requestName, requestOptions }) => {
            return request.exec(requestName, requestOptions)
                .then((response) => {
                    setProgressBarObject(prev => {
                        return {
                            ...prev,
                            successLength: prev.successLength + 1,
                        };
                    });

                    return response;
                })
                .catch((error) => {
                    setProgressBarObject(prev => {
                        return {
                            ...prev,
                            errors: [...prev.errors, { data: prev.errors.length, error: error }],
                        };
                    });
                });
        });
    };

    const setError = (error) => {
        setErrors(errors => [...errors, error]);
    };

    const fetchData = async () => {
        const _response = await requestQueueWithAllSettled(getQueue, setError);
        if (responseHandler) {
            setResponse(responseHandler(_response));
        } else {
            setResponse(_response);
        }
    };

    React.useEffect(() => {
        if (requestQueueOptions && requestShouldFire) {
            setProgressBarObject(prev => {
                return {
                    ...prev,
                    allLength: requestQueueOptions.length,
                };
            });
            setLoading(true);
            setErrors([]);

            fetchData().then(() => {
                setLoading(false);
                fireRequest(false);
            });
        }

        return () => {
            request.abort();
        };
    }, [requestShouldFire]);

    return [isLoading, response, errors, () => {fireRequest(true);}, progressBarObject];
}
