import { Dict } from '../../types';
import { EMPTY_DATA } from '../constants';
import { rubs } from '../utils/rubs';

export enum CAR_STATUS {
    available = 'available',
    service = 'service',
    yaauto_update = 'yaauto_update',
    cleaning_light = 'cleaning_light',
    cleaning = 'cleaning',
    fueling = 'fueling',
    beta_available = 'beta_available',
    for_personal_use = 'for_personal_use',
    new = 'new',
    drive_team_access = 'drive_team_access',
    old_state_parkin = 'old_state_parkin',
    old_state_accept = 'old_state_accept',
    old_state_riding = 'old_state_riding',
    transformation = 'transformation',
    old_state_reserv = 'old_state_reserv',
    old_state_reserv_testing = 'old_state_reservation',
    airport = 'airport',
    no_class = 'no_class',
    fake = 'fake',
    total = 'total'
}

export interface IModel {
    cars_count: number;
    code: string;
    fuel_cap_side: string;
    fuel_icon_url?: string;
    fuel_type: string;
    image_angle_url?: string;
    image_background_url?: string;
    image_large_url: string;
    image_map_url_2x: string;
    image_map_url_3x: string;
    image_pin_url_2x: string;
    image_pin_url_3x: string;
    image_small_url: string;
    manufacturer: string;
    name: string;
    rank?: number;
    registry_manufacturer: string;
    registry_model: string;
    short_name: string;
    specifications: {
        id: string;
        name: string;
        position: number;
        value: string;
    }[];
    visual?: { background: { bottom_color: string; top_color: string }; title: { color: string } }[];
}

interface IFormerNumber {
    timestamp: number;
    number: string;
}

const FRACTION_DIGITS_2 = 2;

export default class CarInfo {
    public static colors: string[];
    public static filters: number[];
    public static fuel_card_number: string;
    public static geocoded_location: string;
    public static id: string;
    public static imei: string;
    public static insurance_agreement_number: string;
    public static insurance_provider: string;
    public static lag: {
        blackbox: number;
        created: number;
        heartbeat: number;
        location: number;
    };
    public static location: {
        areas: {
            id: string;
            tags: string[];
            title: string;
        }[];
        base_latitude: number;
        base_longitude: number;
        base_timestamp: number;
        course: number;
        lat: number;
        lon: number;
        name: string;
        precision: number;
        tags: string[];
        timestamp: number;
        type: string;
    };
    public static locations: {
        course: number;
        latitude: number;
        longitude: number;
        precision: number;
        since: number;
        timestamp: number;
        type: string;
    }[];
    public static model_id: string;
    public static models: {
        cars_count: number;
        code: string;
        fuel_cap_side: string;
        fuel_icon_url: string;
        fuel_type: string;
        image_angle_url: string;
        image_large_url: string;
        image_map_url_2x: string;
        image_map_url_3x: string;
        image_pin_url_2x: string;
        image_pin_url_3x: string;
        image_small_url: string;
        manufacturer: string;
        name: string;
        rank: number;
        registry_manufacturer: string;
        registry_model: string;
        short_name: string;
        specifications: Dict<any> [];
        visual: Dict<any>;
    }[];
    public static offer_name?: string;
    public static former_numbers: IFormerNumber[];
    public static number: string;
    public static patches: number[];
    public static registration_id: string;
    public static responsible_picker: string;
    public static session_info: {
        meta: Dict<any>;
        session: Dict<any>;
    };
    public static sf: {
        comment: string;
        display_name: string;
        index: number;
        is_important: boolean;
        name: string;
        public_icon: string;
        tag_flow: string;
        tag_flow_priority: number;
    }[];
    public static status: string;
    public static tags: {
        display_name: string;
        object_id: string;
        priority: number;
        tag: string;
        tag_id: string;
    }[];
    public static telematics: {
        acc_voltage: number;
        acc_voltage_updated_at: number;
        ext_voltage: number;
        ext_voltage_updated_at: number;
        fuel_level: number;
        fuel_level_updated_at: number;
        horn_dout3: any;
        is_engine_on: boolean;
        is_engine_on_updated_at: number;
        is_front_left_door_open: boolean;
        is_front_left_door_open_updated_at: number;
        is_front_right_door_open: boolean;
        is_front_right_door_open_updated_at: number;
        is_hood_open: boolean;
        is_hood_open_updated_at: number;
        is_rear_left_door_open: boolean;
        is_rear_left_door_open_updated_at: number;
        is_rear_right_door_open: boolean;
        is_rear_right_door_open_updated_at: number;
        is_trunk_open: boolean;
        is_trunk_open_updated_at: number;
        low_battery_marks: any;
        mark_1: any;
        mileage: number;
        mileage_updated_at: number;
        speed: number;
        speed_updated_at: number;
        visible_marks: any;
        second_fuel_type?: string;
        second_fuel_level?: number;
    };
    public static update_timestamp: number;
    public static usage: string;
    public static user: Dict<any>;
    public static vin: string;
    public static _tags?: Dict<object>;
    public static documents?: Dict<any>;
}

export class CarInfoHandler extends CarInfo {
    public static getUsage(): string | null {
        return this.usage;
    }

    public static getModelName(): string {
        const model = this?.models?.[0] || {};

        return model.name || model.code || this?.model_id || EMPTY_DATA;
    }

    public static getId(): string {
        return this
            && this.id
            || EMPTY_DATA;
    }

    public static getModelNameManufacturer(): string {
        return this
            && this.models && this.models[0]
            && this.models[0].manufacturer && this.models[0].short_name
            && this.models[0].manufacturer + ' ' + this.models[0].short_name
            || EMPTY_DATA;
    }

    public static getModelFullName(): string {
        return this?.models?.[0]?.name || EMPTY_DATA;
    }

    public static getNumber(): string {
        return this
            && this.number
            || EMPTY_DATA;
    }

    public static getFormerNumbers(): IFormerNumber[] {
        return this?.former_numbers
            ?.sort((a, b) => b.timestamp - a.timestamp)
            .filter((el => {
                return el.number !== this.number;
            }))
            || [];
    }

    public static getStatus(): string {
        return this
            && this.status
            || EMPTY_DATA;
    }

    public static getImei(): string {
        return this
            && this.imei
            || EMPTY_DATA;
    }

    public static getSts(): string {
        return this
            && this.registration_id
            || EMPTY_DATA;
    }

    public static getVin(): string {
        return this
            && this.vin
            || EMPTY_DATA;
    }

    public static getSessionMeta(): Dict<any> | null {
        return this
            && this.session_info
            && this.session_info.meta
            || null;
    }

    public static getAccLevel(): number | null {
        return (this?.telematics?.acc_voltage &&
            +this.telematics.acc_voltage.toFixed(FRACTION_DIGITS_2)) ?? null;
    }

    public static getExtLevel(): number | null {
        return (this?.telematics?.ext_voltage &&
            +this.telematics.ext_voltage.toFixed(FRACTION_DIGITS_2)) ?? null;
    }

    public static getFuelType(): string | null {
        return this
            && this.models
            && this.models[0]
            && this.models[0].fuel_type || null;
    }

    public static getSecondFuelType(): string | null {
        return this?.telematics?.second_fuel_type || null;
    }

    public static getFuelLevel(): number | null {
        return (this?.telematics?.fuel_level &&
            +this.telematics.fuel_level.toFixed(FRACTION_DIGITS_2)) ?? null;
    }

    public static getSecondFuelLevel(): number | null {
        return this?.telematics?.second_fuel_level ?? null;
    }

    public static getUserInfo(): Dict<any> | null {
        return this
            && this.user
            || null;
    }

    public static isManualGearbox(): boolean {
        const MANUAL_GEARBOX = 'manual_gearbox';

        return this
            && this.tags
            && Array.isArray(this.tags)
            && this.tags.some((_t: any) => _t.tag === MANUAL_GEARBOX);
    }

    public static isEngineOn(): boolean {
        return this
            && this.telematics
            && this.telematics.is_engine_on;
    }

    public static getMileage(): number | null {
        return +this?.telematics?.mileage?.toFixed(1) ?? null;
    }

    public static getSpeed(): number | null {
        return +this?.telematics?.speed?.toFixed(1) ?? null;
    }

    public static getSessionCurrentOffer() {
        return this
            && this.session_info
            && this.session_info.session
            && this.session_info.session.specials
            && this.session_info.session.specials.current_offer
            || null;
    }

    public static getInsuranceType() {
        const current_offer = CarInfoHandler.getSessionCurrentOffer.call(this);

        return current_offer?.prices?.insurance_type || null;
    }

    public static getInsuranceProvider() {
        return this.insurance_provider || null;
    }

    public static getSessionTotalPrice() {
        return rubs(this?.session_info?.session?.specials?.total_price) ?? null;
    }

    public static getLocationLonLat() {
        return this
            && this.location
            && this.location.lon
            && this.location.lat
            && [
                this.location.lon,
                this.location.lat,
            ]
            || null;
    }

    public static getDistrict(): string {
        let area = '';

        this.location?.areas?.reduce((prev: any, cur: any) => {
            if (cur?.tags?.indexOf('techzone') > -1) {
                area = cur.title;
            }

            return `${prev} ${area}`;
        }, '');

        return area?.trim();
    }

    public static isFrontLeftDoorOpen(): boolean {
        return this
            && this.telematics
            && this.telematics.is_front_left_door_open;
    }

    public static isFrontRightDoorOpen(): boolean {
        return this
            && this.telematics
            && this.telematics.is_front_right_door_open;
    }

    public static isRearLeftDoorOpen(): boolean {
        return this
            && this.telematics
            && this.telematics.is_rear_left_door_open;
    }

    public static isRearRightDoorOpen(): boolean {
        return this
            && this.telematics
            && this.telematics.is_rear_right_door_open;
    }

    public static isTrunkOpen(): boolean {
        return this
            && this.telematics
            && this.telematics.is_trunk_open;
    }

    public static isHoodOpen(): boolean {
        return this
            && this.telematics
            && this.telematics.is_hood_open;
    }

    public static getOsagoLink(): string {
        return this?.documents?.find(el => el.title === 'ОСАГО')?.link ?? '';
    }
}

export interface IConfig {
    finishOrder: boolean;
    ignoreError: boolean;
    callback: () => void;
    question: string;
    actionName?: string;
    user_id?: string;
    evolution_mode?: string;
    car_id?: string;
}

export const CAR_ACTIONS: any = {
    OPEN: 'OPEN',
    HORN: 'HORN',
    LONG_HORN_ENABLE: 'SCENARIO_ENABLE_LONGHORN', //command= //SET_PARAM&id=1063&value=1
    LONG_HORN_DISABLE: 'SCENARIO_DISABLE_LONGHORN', //command= //SET_PARAM&id=1063&value=0
    SCENARIO_POLITE_RESTART: 'SCENARIO_POLITE_RESTART',
    OPEN_DOORS: 'OPEN_DOORS',
    CLOSE_DOORS: 'CLOSE_DOORS',
    YADRIVE_UNLOCK_HOOD: 'YADRIVE_UNLOCK_HOOD',
    YADRIVE_LOCK_HOOD: 'YADRIVE_LOCK_HOOD',
    SCENARIO_BLE_RESET_FORCED_END_OF_LEASE: 'SCENARIO_BLE_RESET_FORCED_END_OF_LEASE',
    SCENARIO_POLITE_FORCED_END_OF_LEASE: 'SCENARIO_POLITE_FORCED_END_OF_LEASE',
    YADRIVE_END_OF_LEASE: 'YADRIVE_END_OF_LEASE',
    YADRIVE_START_OF_LEASE: 'YADRIVE_START_OF_LEASE',
    YADRIVE_WARMING: 'YADRIVE_WARMING',

    UNLOCK_CHARGE_PORT: 'unlock-charge-port',
    UNLOCK_CHARGE_CONNECTOR: 'unlock-charge-connector',
};

export const SCOOTER_ACTIONS: any = {
    SCENARIO_UNLOCK_DOORS_AND_HOOD: 'SCENARIO_UNLOCK_DOORS_AND_HOOD',
    YADRIVE_UNLOCK_HOOD: 'YADRIVE_UNLOCK_HOOD',
    OPEN_DOORS: 'OPEN_DOORS',
    CLOSE_DOORS: 'CLOSE_DOORS',
    HORN_AND_BLINK: 'HORN_AND_BLINK',
};
