import React, { ReactElement } from 'react';

import { AccountType, PaymentSystem } from '../components/Settings/Wallets/types';
import { CITIES, EMPTY_DATA, ONE_SECOND } from '../constants';
import FormatDate, { durationBetween, FormatDateInString } from '../ui/FormatDate';
import { DISPLAY_TEXT_KEY, IOptionInfo } from '../ui/Select';
import { rubs } from '../utils/rubs';
import { UserInfoHandler, UserSetup } from './user';

export interface ICar {
    number: string;
    model_id: string;
    id: string;
    vin?: string;
}

export interface IModelItem {
    name: string;
    code: string;
}

interface IModel {
    [key: string]: IModelItem;
}

export enum SESSION_TYPE {
    RIDING_COMPILATION = 'riding_compilation',
    EVENTS_SESSION = 'events_session'
}

export interface IOfferProto {
    Name: string;
    SelectedCharge: AccountType | PaymentSystem;
    Corrector: string[];
    ShortName: string;
    GroupName: string;
    CashbackPercent: number;
    ChargableAccounts: string[];
    ObjectId: string;
    ConstructorId: string;
    UserId: string;
    ParentId: string;
    DeviceIdAccept: string;
    PackOffer: any;
    FixPointOffer: any;
    StandartOffer: any;
    FromScanner: any;
    TransferredFrom?: any;
    TransferType?: any;
    Origin: Origin;
    LongTermOffer?: any;
    Marker?: string;
}

interface ISessionSegmentEvent {
    timestamp: number;
    tag_name: string;
    action: string;
    event_id: string;
    user_id: string;
    transformation_skipped_by_external_command?: boolean;
}

export interface ISessionBill {
    type: string;
    title: string;
    cost: number;
    details?: string;
}

export interface ISessionCurrentOffer {
    name: string;
    group_name: string;
    switchable: boolean;
    short_description: string[];
    short_name: string;
    discounts: { discount: number; title: string }[];
    offer_id: string;
    wallets: { id: string; selected?: boolean }[];
    type: string;
    summary_discount: any;
    user_proposition: any;
    prices: any;
    rerun_price_km: any;
    free: any;
}

interface IDeviceDiffPoint {
    timestamp: number;
    longitude: number;
    latitude: number;
    since: number;
    course: number;
    type: string;
}

interface IDeviceDiff {
    finish: IDeviceDiffPoint;
    hr_mileage: string;
    mileage: number;
    start: IDeviceDiffPoint;
}

interface ISessionSegmentSession {
    specials: {
        current_offer_state: ICurrentOfferState;
        current_offer: ISessionCurrentOffer;
        bill: ISessionBill[];
        total_price_hr: string;
        total_duration: number;
    };
}

interface ICurrentOfferState {
    current_pack_price: number;
    distance_threshold_push_sent: number;
    duration_threshold_push_sent: number;
    exceeded: boolean;
    overrun_distance: number;
    overrun_time: number;
    pack_price: number;
    pack_price_round: number;
    payments: any[];
    remaining_distance: number;
    remaining_time: number;
    return_cost: number;
    return_fee: number;
    return_overrun_cost: number;
    return_pack_price: number;
    show_remainders: boolean;
    since: number;
    until: number;
}

interface ISessionSegmentMetaBasic {
    start: number;
    finish: number;
    session_id: string;
    total_price: number;
    total_duration: number;
}

interface ISessionSegmentMeta extends ISessionSegmentMetaBasic {
    finished: boolean;
    corrupted: boolean;
    user_id: string;
    object_id: string;
    inconsistency: boolean;
}

interface ISessionSegment extends ISessionSegmentMetaBasic {
    events: ISessionSegmentEvent[];
    session: ISessionSegmentSession;
    meta: ISessionSegmentMeta;
    __type: SESSION_TYPE;
    delegation_type?: DELEGATION_TYPE;
    expected_finish?: number;
}

export interface InsuranceNotification {
    created: number;
    finish: number;
    id: number;
    sent: number;
    start: number;
}

export interface ISessionInfo {
    car: ICar;
    device_diff: IDeviceDiff;
    trace_tags: any[];
    offer_proto: IOfferProto;
    user_details: {
        last_name: string;
        first_name: string;
        pn: string;
        status: string;
        username: string;
        id: string;
        setup: UserSetup;
    };
    insurance_notifications: InsuranceNotification[];
    segment: ISessionSegment;
    start_geo_tags: string[];
    geo_tags: string[];
    futures_offer?: any;
}

export default class SessionsHistoryItem {
    public static sessions: ISessionInfo[];
    public static has_more: boolean;
    public static models: IModel;
}

export const fee_drop_zone_max = 'fee_drop_zone_max';
export const total = 'total';
const sliceEnd = 3;

export enum INSURANCE_TYPE {
    standart = 'Базовая страховка',
    full = 'Полная страховка'
}

export enum Origin {
    TAXI = 'taxi'
}

export enum DELEGATION_TYPE {
    'free' = 'free',
    'p2p' = 'p2p',
    'p2p_pass_offer' = 'p2p_pass_offer',
    'service' = 'service'
}

type I_DELEGATION_TYPE_DESCRIPTION = {
    [key in DELEGATION_TYPE]: string
}
type I_DELEGATION_INDEX = {
    [key: number]: DELEGATION_TYPE;
}

export const DELEGATION_INDEX: I_DELEGATION_INDEX = {
    0: DELEGATION_TYPE.free,
    1: DELEGATION_TYPE.p2p,
    2: DELEGATION_TYPE.p2p_pass_offer,
    3: DELEGATION_TYPE.service,
};
export const DELEGATION_TYPE_DESCRIPTION: I_DELEGATION_TYPE_DESCRIPTION = {
    [DELEGATION_TYPE.free]: 'подвешенная аренда',
    [DELEGATION_TYPE.p2p]: 'передача руля без передачи остатка',
    [DELEGATION_TYPE.p2p_pass_offer]: 'передача руля с остатком пакета',
    [DELEGATION_TYPE.service]: 'сервисная передача для заправки',
};

export class SessionHistoryInfoHandler extends SessionsHistoryItem {

    public static hasMore() {
        return this.has_more;
    }

    public static getData() {
        return this;
    }

    public static getCount() {
        return this && this.sessions && this.sessions.length;
    }

    public static buildOptions(descriptionTemplate: (props) => ReactElement): IOptionInfo[] {
        return this
            && this.sessions
            && this.sessions.map((session, index) => {
                const text = <>
                    <FormatDate value={SessionHistoryInfoHandler.getStart.call(this, index)}/>
                    <span> {EMPTY_DATA} </span>
                    <FormatDate value={SessionHistoryInfoHandler.getFinish.call(this, index)} withoutYear/>
                </>;
                const car: ICar = SessionHistoryInfoHandler.getCar.call(this, index);
                const model = SessionHistoryInfoHandler.getModelName.call(this, index);
                const total = SessionHistoryInfoHandler.getTotalPrice.call(this, index);

                return {
                    text,
                    value: index,
                    [DISPLAY_TEXT_KEY]: `${FormatDateInString({
                        value: SessionHistoryInfoHandler.getStart.call(this, index),
                    })} `
                    + `${EMPTY_DATA} `
                    + `${FormatDateInString({
                        value: SessionHistoryInfoHandler.getFinish.call(this, index),
                        withoutYear: true,
                    })}`,
                    description: descriptionTemplate({ model, total, number: car.number }),
                };
            }) || [] as IOptionInfo[];
    }

    public static getCar(item = 0): ICar {
        return this?.sessions?.[item]?.car || {} as ICar;
    }

    public static getInsuranceNotifications(item = 0): InsuranceNotification[] {
        return this?.sessions?.[item]?.insurance_notifications || [];
    }

    public static getCarNumber(item = 0): string {
        const car = SessionHistoryInfoHandler.getCar.call(this, item) || {} as ICar;

        return car && car.number || EMPTY_DATA;
    }

    public static getShortNumber(item = 0): string {
        const car = SessionHistoryInfoHandler.getCar.call(this, item) || {} as ICar;

        return car && car.number && car.number.slice(0, sliceEnd);
    }

    public static getCarId(item = 0): string | null {
        const car = SessionHistoryInfoHandler.getCar.call(this, item) || {} as ICar;

        return car && car.id || null;
    }

    public static getInsurance(item = 0): string | null {
        const offer_proto = SessionHistoryInfoHandler.getOffer.call(this, item);

        return offer_proto?.StandartOffer?.InsuranceType;
    }

    public static getBeforeEvolution(item = 0): string | null {
        const offer_proto = SessionHistoryInfoHandler.getOffer.call(this, item);

        return offer_proto?.ParentId;
    }

    public static getTransferredFrom(item = 0): string | null {
        const offer_proto = SessionHistoryInfoHandler.getOffer.call(this, item);

        return offer_proto?.TransferredFrom;
    }

    public static getTransferredType(item = 0): number | null {
        const offer_proto = SessionHistoryInfoHandler.getOffer.call(this, item);

        return offer_proto?.TransferType;
    }

    public static isContainsBluetoothAction(item = 0): number | null {
        const events = SessionHistoryInfoHandler.getEvents.call(this, item);

        return events.some(e => e.transformation_skipped_by_external_command);
    }

    public static isTaxi(item = 0): boolean {
        const offer_proto = SessionHistoryInfoHandler.getOffer.call(this, item);

        return offer_proto?.Origin === Origin.TAXI;
    }

    public static isLongTerm(item = 0): boolean {
        const offer_proto = SessionHistoryInfoHandler.getOffer.call(this, item);

        return offer_proto?.LongTermOffer;
    }

    public static getDelegationType(item = 0): DELEGATION_TYPE | null {
        const segment = SessionHistoryInfoHandler.getSegment.call(this, item);
        if (segment.__type === SESSION_TYPE.RIDING_COMPILATION) {
            return segment.delegation_type;
        }

        if (segment.__type === SESSION_TYPE.EVENTS_SESSION) {
            return segment?.session?.specials?.delegation_type;
        }

        return null;
    }

    public static getModelName(item = 0) {
        const car = SessionHistoryInfoHandler.getCar.call(this, item);

        return car
            && car.model_id
            && this
            && this.models
            && this.models[car && car.model_id]
            && this.models[car && car.model_id].name;
    }

    public static getLastStatus(item = 0) {
        const events = SessionHistoryInfoHandler.getEvents.call(this, item);
        const lastEvent = events[events.length - 1];

        return lastEvent && lastEvent.tag_name || EMPTY_DATA;
    }

    public static getOffer(item = 0) {
        return this
            && this.sessions
            && this.sessions[item]
            && this.sessions[item].offer_proto;
    }

    public static getOfferName(item = 0) {
        const offer = SessionHistoryInfoHandler.getOffer.call(this, item) || {} as IOfferProto;

        return offer?.Name !== offer?.GroupName
            ? (`${offer?.Name} ${offer?.GroupName ? `(${offer?.GroupName})` : ''}`)
            : offer?.Name || EMPTY_DATA;
    }

    public static getMeta(item = 0): ISessionSegmentMeta {
        return this
            && this.sessions
            && this.sessions[item]
            && this.sessions[item].segment
            && this.sessions[item].segment.meta;
    }

    public static getUserId(item = 0): string | null {
        const meta = SessionHistoryInfoHandler.getMeta.call(this, item);
        const details = SessionHistoryInfoHandler.getUserDetails.call(this, item);

        return meta
            && meta.user_id
            || details?.id;
    }

    public static getUserPrintName(item = 0): string | null {
        const user = SessionHistoryInfoHandler.getUserDetails.call(this, item) || {};

        return UserInfoHandler.getPrintName.call(user);
    }

    public static getSegment(item = 0): ISessionSegment {
        return this
            && this.sessions
            && this.sessions[item]
            && this.sessions[item].segment || {} as ISessionSegment;
    }

    public static getSessionId(item = 0) {
        const segment: ISessionSegment = SessionHistoryInfoHandler.getSegment.call(this, item);
        if (segment.__type === SESSION_TYPE.RIDING_COMPILATION) {
            return segment.session_id;
        }

        if (segment.__type === SESSION_TYPE.EVENTS_SESSION) {
            return segment.meta && segment.meta.session_id;
        }

        return 0;
    }

    public static getStart(item = 0) {
        const segment: ISessionSegment = SessionHistoryInfoHandler.getSegment.call(this, item);
        if (segment.__type === SESSION_TYPE.RIDING_COMPILATION) {
            return segment.start * ONE_SECOND;
        }

        if (segment.__type === SESSION_TYPE.EVENTS_SESSION) {
            return segment.meta && segment.meta.start * ONE_SECOND;
        }

        return 0;
    }

    public static getFinish(item = 0) {
        const segment: ISessionSegment = SessionHistoryInfoHandler.getSegment.call(this, item);
        if (segment.__type === SESSION_TYPE.RIDING_COMPILATION) {
            return segment.finish * ONE_SECOND;
        }

        if (segment.__type === SESSION_TYPE.EVENTS_SESSION) {
            return segment.meta && segment.meta.finish && segment.meta.finish * ONE_SECOND;
        }

        return 0;
    }

    public static getExpectedFinish(item = 0) {
        const segment: ISessionSegment = SessionHistoryInfoHandler.getSegment.call(this, item);

        return segment.expected_finish && segment.expected_finish * ONE_SECOND;
    }

    public static getUserDetails(item = 0) {
        return this
            && this.sessions
            && this.sessions[item]
            && this.sessions[item].user_details;
    }

    public static getCityForIncident(item = 0) {
        const start_city = this
            && this.sessions
            && this.sessions[item]
            && this.sessions[item].start_geo_tags
            && this.sessions[item].start_geo_tags[0];

        return CITIES?.[start_city]?.name;
    }

    public static getCity(item = 0) {
        const start_city = this
            && this.sessions
            && this.sessions[item]
            && this.sessions[item].start_geo_tags
            && this.sessions[item].start_geo_tags[0];
        const finish_city = this
            && this.sessions
            && this.sessions[item]
            && this.sessions[item].geo_tags
            && this.sessions[item].geo_tags[0];

        if (start_city === finish_city || (!start_city && finish_city)) {
            return CITIES?.[finish_city]?.short_name2 || CITIES?.[finish_city]?.name;
        }

        return `${CITIES?.[start_city]?.short_name2 || CITIES?.[start_city]?.name || EMPTY_DATA} ` +
            `⟶ ${CITIES?.[finish_city]?.short_name2 || CITIES?.[finish_city]?.name || EMPTY_DATA}`;
    }

    public static getDuration(item = 0) {
        const start = SessionHistoryInfoHandler.getStart.call(this, item);
        let finish = SessionHistoryInfoHandler.getFinish.call(this, item);

        if (start && !finish) {
            finish = Date.now();
        }

        return start && finish && durationBetween([finish / ONE_SECOND, start / ONE_SECOND]) || EMPTY_DATA;
    }

    public static getOfferDetails(item = 0): string {
        const segment = SessionHistoryInfoHandler.getSegment.call(this, item);

        return segment?.session?.specials?.current_offer?.detailed_description || segment?.offer?.detailed_description;
    }

    public static getServiceItems(item = 0){
        const segment = SessionHistoryInfoHandler.getSegment.call(this, item);

        return segment?.session?.specials?.servicing_info || [];
    }

    public static getBill(item = 0) {
        const segment = SessionHistoryInfoHandler.getSegment.call(this, item);
        if (segment.__type === SESSION_TYPE.RIDING_COMPILATION) {
            return segment && segment.bill;
        }

        if (segment.__type === SESSION_TYPE.EVENTS_SESSION) {
            return segment
                && segment.session
                && segment.session.specials
                && segment.session.specials.bill;
        }

        return 0;
    }

    public static getServicingDuration(item = 0) {
        const state = SessionHistoryInfoHandler.getSessionState.call(this, item);

        return state?.servicing_duration;
    }

    public static getServicingOmittedPrice(item = 0) {
        const state = SessionHistoryInfoHandler.getSessionState.call(this, item);

        return state?.servicing_omitted_price || 0;
    }

    public static getCurrentOffer(item = 0) {
        const segment = SessionHistoryInfoHandler.getSegment.call(this, item);
        if (segment.__type === SESSION_TYPE.RIDING_COMPILATION) {
            return segment && segment.offer;
        }

        if (segment.__type === SESSION_TYPE.EVENTS_SESSION) {
            return segment
                && segment.session
                && segment.session.specials
                && segment.session.specials.current_offer;
        }

        return 0;
    }

    public static getTotalPrice(item = 0): string {
        const segment = SessionHistoryInfoHandler.getSegment.call(this, item);
        if (segment.__type === SESSION_TYPE.RIDING_COMPILATION) {
            return (segment.total_price || segment.total_price == 0) && rubs(segment.total_price);
        }

        if (segment.__type === SESSION_TYPE.EVENTS_SESSION) {
            return segment
                && segment.session
                && segment.session.specials
                && (segment.session.specials.total_price || segment.session.specials.total_price == 0)
                && rubs(segment.session.specials.total_price);
        }

        return EMPTY_DATA;
    }

    public static getDeviceDiff(item = 0): IDeviceDiff {
        return this
            && this.sessions
            && this.sessions[item]
            && this.sessions[item].device_diff || {};
    }

    public static getLatitudeDiff(item = 0) {
        const device_diff = SessionHistoryInfoHandler.getDeviceDiff.call(this, item);

        return device_diff
            && device_diff.finish
            && device_diff.finish.latitude
            && device_diff.start
            && device_diff.start.latitude
            && (device_diff.finish.latitude - device_diff.start.latitude);
    }

    public static getLongitudeDiff(item = 0) {
        const device_diff = SessionHistoryInfoHandler.getDeviceDiff.call(this, item);

        return device_diff
            && device_diff.finish
            && device_diff.finish.longitude
            && device_diff.start
            && device_diff.start.longitude
            && (device_diff.finish.longitude - device_diff.start.longitude);
    }

    public static getCoordinatesDiff(item = 0): boolean {
        const latitudeDiff = SessionHistoryInfoHandler.getLatitudeDiff.call(this, item);
        const longitudeDiff = SessionHistoryInfoHandler.getLongitudeDiff.call(this, item);

        return latitudeDiff !== 0 || longitudeDiff !== 0;
    }

    public static getMileage(item = 0) {
        const device_diff = SessionHistoryInfoHandler.getDeviceDiff.call(this, item);

        return device_diff
            && device_diff.mileage;
    }

    public static getHrMileage(item = 0) {
        const device_diff = SessionHistoryInfoHandler.getDeviceDiff.call(this, item);

        return device_diff
            && device_diff.hr_mileage;
    }

    public static isSessionCompiled(item = 0): boolean {
        const segment = SessionHistoryInfoHandler.getSegment.call(this, item);

        return segment.__type === SESSION_TYPE.RIDING_COMPILATION;
    }

    public static getTraceTags(item = 0) {
        return this
            && this.sessions
            && this.sessions[item]
            && this.sessions[item].trace_tags;
    }

    public static isOfferCorp(item = 0) {
        const offer_proto = SessionHistoryInfoHandler.getOffer.call(this, item);

        return offer_proto
        && offer_proto.SelectedCharge
        && (offer_proto.SelectedCharge !== AccountType.CARD && offer_proto.SelectedCharge !== PaymentSystem.MOBILE)
            ? offer_proto.SelectedCharge
            : null;
    }

    public static isOfferYAC(item = 0) {
        const offer_proto = SessionHistoryInfoHandler.getOffer.call(this, item);

        return offer_proto?.SelectedCharge === PaymentSystem.YANDEX_ACCOUNT;
    }

    public static getSelectedCharge(item = 0) {
        const offer_proto = SessionHistoryInfoHandler.getOffer.call(this, item);

        return offer_proto?.SelectedCharge;
    }

    public static idFeeDropZone(item = 0) {
        const bill = SessionHistoryInfoHandler.getBill.call(this, item);

        return bill?.some?.(el => el.type === fee_drop_zone_max);
    }

    public static getFinalCost(item = 0) {
        const bill = SessionHistoryInfoHandler.getBill.call(this, item);

        return bill?.find?.(el => el.type === total)?.cost ?? 0;
    }

    public static getEvents(item = 0) {
        const segment = SessionHistoryInfoHandler.getSegment.call(this, item);

        return segment && segment.events || [];
    }

    public static getLastEventsStatus(item = 0) {
        const segment = SessionHistoryInfoHandler.getSegment.call(this, item);

        return segment?.events?.[segment?.events?.length - 1]?.tag_name;
    }

    public static getSession(item = 0) {
        return this
            && this.sessions
            && this.sessions[item];
    }

    public static getRideCost(item = 0) {
        const currentOffer = SessionHistoryInfoHandler.getCurrentOffer.call(this, item);

        return currentOffer
            && currentOffer.prices.riding
            && rubs(currentOffer.prices.riding);
    }

    public static getParkingCost(item = 0) {
        const currentOffer = SessionHistoryInfoHandler.getCurrentOffer.call(this, item);

        return currentOffer
            && currentOffer.prices.parking
            && rubs(currentOffer.prices.parking);
    }

    public static getFuturesOffer(item = 0) {
        return this
            && this.sessions
            && this.sessions[item]
            && this.sessions[item].futures_offer
            || null;
    }

    public static isFixOffer(item = 0): boolean {
        const currentOffer = SessionHistoryInfoHandler.getCurrentOffer.call(this, item);

        return currentOffer?.type === 'fix_point';
    }

    public static isSessionFinished(item = 0): boolean {
        const finish = SessionHistoryInfoHandler.getFinish.call(this, item);

        return !!finish;
    }

    public static getSessionState(item = 0): ICurrentOfferState {
        const segment = SessionHistoryInfoHandler.getSegment.call(this, item);

        return segment?.session?.specials?.current_offer_state;
    }

    public static getVoyageId(item = 0): boolean {
        const offer_proto = SessionHistoryInfoHandler.getOffer.call(this, item);

        return offer_proto?.Marker;
    }

    public static getFleetMark(item = 0, tag: string): boolean {
        const session = SessionHistoryInfoHandler.getSession.call(this, item);

        return session?.offer_groupping_tags?.includes(tag);
    }
}
