import { Dict, ITagItem } from '../../types';
import { EMPTY_DATA, ONE_SECOND } from '../constants';
import { FormatDateInString } from '../ui/FormatDate';
import { rubs } from '../utils/rubs';

export class UserSetup {
    'email': {
        'address': string;
        'verified': boolean;
    };
    'phone': {
        'number': string;
        'verified': boolean;
    };
}

class Registration {
    'approved_at': number;
    'joined_at': number;
}

class UserBillingInfo {
    bonus: number;
    task_count: number;
    debt: number;
    yandex_account: {
        balance: number;
    };
}

export const GENDER = {
    'МУЖ': 'male',
    'ЖЕН': 'female',
};

export class MinimalUserInfo {
    public static last_name: string;
    public static first_name: string;
    public static pn: string;
    public static id: string;
    public static username: string;
}

export class BasicUserInfo extends MinimalUserInfo {
    public static 'setup': UserSetup;
    public static 'status': string;
    public static 'tags': ITagItem[];
    public static 'preliminary_payments': {
        'amount': number;
        'enabled': boolean;
    };
    public static blackbox: {
        plus: boolean;
        yandexoid: boolean;
    };
}

export default class UserInfo extends BasicUserInfo {
    public static documents: {
        driving_license: {
            'birth_date': string;
            'categories': string;
            'categories_b_valid_to_date': string;
            'experience_from': string;
            'first_name': string;
            'issue_date': string;
            'last_name': string;
            'middle_name': string;
            'number_back': string;
            'number_front': string;
            'revision': string;
        }[];
        passport: {
            'birth_date': string;
            'birth_place': string;
            'citizenship': string;
            'doc_type': string;
            'doc_value': string;
            'first_name': string;
            'gender': string;
            'issue_date': string;
            'last_name': string;
            'middle_name': string;
            'registration_apartment': string;
            'registration_area': string;
            'registration_house': string;
            'registration_locality': string;
            'registration_street': string;
            'subdivision_code': string;
            'revision': string;
        }[];
    };
    public static 'driving_license_revision': string;
    public static 'is_first_riding': boolean;
    public static 'passport_revision': string;
    public static 'is_mechanic_transmission_allowed': boolean;
    public static registration_state: Dict<any>;
    public static registration_chat: Dict<string>;
    public static billing: UserBillingInfo;
    public static uid: string;
    public static photos: Dict<any>;
    public static registration: Registration;
    public static realtime_data: {
        experiments?: number[];
    };
}

const PASSPORT_SERIES_LIMIT = 4;
const PASSPORT_SERIES_FIRST_LIMIT = 2;

export class UserInfoHandler extends UserInfo {
    public static getEmail(): string {
        return this
            && this.setup
            && this.setup.email
            && this.setup.email.address || EMPTY_DATA;
    }

    public static getUid(): string {
        return this
            && this.uid || EMPTY_DATA;
    }

    public static getId(): string {
        return this
            && this.id;

    }

    public static getStatus(): string {
        return this
            && this.status;
    }

    public static getBonuses(): string {
        return this?.billing?.bonus
            ? rubs(this.billing.bonus)
            : this?.billing?.bonus === 0
                ? '0.00₽'
                : EMPTY_DATA;
    }

    public static getYaAccount(): string {
        return rubs(this?.billing?.yandex_account?.balance || 0);
    }

    public static getDebt(): string {
        return this
            && this.billing
            && this.billing.debt
            && rubs(this.billing.debt)
            || EMPTY_DATA;
    }

    public static getExperiments(): number[] {
        return this
            && this.realtime_data
            && this.realtime_data.experiments
            || [];
    }

    public static getPhone(): string {
        return this
            && this.setup
            && this.setup.phone
            && this.setup.phone.number
            //&& formatPhone(this.setup.phone.number)
            || EMPTY_DATA;
    }

    public static getSimplePhone(): string {
        return this
            && this.setup
            && this.setup.phone
            && this.setup.phone.number
            || EMPTY_DATA;
    }

    public static isPlus() {
        return this?.blackbox?.plus || false;
    }

    public static getBirthDay() {
        const passport = UserInfoHandler.getPassportData.call(this);

        return passport
            && passport.birth_date
            || EMPTY_DATA;
    }

    public static getRegistration() {
        return this
            && this.registration
            && this.registration.approved_at * ONE_SECOND
            || EMPTY_DATA;
    }

    public static getRegistrationState() {
        return this
            && this.registration_state
            && this.registration_state.chat_action_id
            || EMPTY_DATA;
    }

    public static getRegistrationChatStage() {
        return this?.registration_chat?.stage || EMPTY_DATA;
    }

    public static getJoined() {
        return this
            && this.registration
            && this.registration.approved_at
            && FormatDateInString({
                value: new Date(this.registration.joined_at * ONE_SECOND),
            });
    }

    public static getBirthPlace() {
        const passport = UserInfoHandler.getPassportData.call(this);

        return passport
            && passport.birth_place
            || EMPTY_DATA;
    }

    public static getRegistrationPlace() {
        const passport = UserInfoHandler.getPassportData.call(this);

        return passport
            && `${passport.registration_region && passport.registration_region
            || EMPTY_DATA} ${passport.registration_locality && passport.registration_locality
            || EMPTY_DATA}${passport.registration_area && ` ,${passport.registration_area}`
            || EMPTY_DATA} ${passport.registration_street && ` ,${passport.registration_street}`
            || EMPTY_DATA} ${passport.registration_house && passport.registration_house
            || EMPTY_DATA} ${passport.registration_housing && `/ ${passport.registration_housing}`
            || EMPTY_DATA}${passport.registration_apartment && ` - ${passport.registration_apartment}`
            || EMPTY_DATA}` || EMPTY_DATA;
    }

    public static getRegistrationDateUntil() {
        const passport = UserInfoHandler.getPassportData.call(this);

        return passport && passport.registration_expiration_date || EMPTY_DATA;
    }

    public static getUsername() {
        return this
            && this.username
            || EMPTY_DATA;
    }

    public static getTags() {
        return this
            && this.tags
            || [];
    }

    public static getPrintName() {
        const { last_name = '', first_name = '', pn = '', username = '' } = this || {};

        return this
        && !last_name && !first_name && !pn
            ? username
            : `${last_name} ${first_name} ${pn}`;
    }

    public static getShortPrintName() {
        const { last_name = '', first_name = '', pn = '', username = '' } = this || {};

        return this
        && !last_name && !first_name && !pn
            ? username
            : `${last_name ? last_name : EMPTY_DATA} `
            + `${first_name ? `${first_name[0]}.` : ''} `
            + `${pn ? `${pn[0]}.` : ''}`;
    }

    public static getPrintNameWithId() {
        const { last_name = '', first_name = '', pn = '', username = '', id = '' } = this || {};

        return this
        && last_name && first_name && pn
            ? `${last_name} ${first_name} ${pn}` :
            username
                ? username
                : id;
    }

    public static getPrintNameWithoutPn() {
        const { last_name = '', first_name = '', username = '' } = this || {};

        return this
        && !last_name && !first_name
            ? username
            : `${last_name} ${first_name}`;
    }

    public static getPrintNameWithoutUserName() {
        const { last_name = '', first_name = '', pn = '' } = this || {};

        return this
            && `${last_name} ${first_name} ${pn}` || '';
    }

    public static getName() {
        return this.first_name;
    }

    public static getPassportName() {
        const passport = UserInfoHandler.getPassportData.call(this);

        return passport?.first_name;
    }

    public static getLastName() {
        return this.last_name;
    }

    public static getPassportLastName() {
        const passport = UserInfoHandler.getPassportData.call(this);

        return passport?.last_name;
    }

    public static getPn() {
        return this.pn;
    }

    public static getPassportMiddleName() {
        const passport = UserInfoHandler.getPassportData.call(this);

        return passport?.middle_name;
    }

    public static isPhoneVerified() {
        return this
            && this.setup
            && this.setup.phone
            && this.setup.phone.verified || false;
    }

    public static getPassportData() {
        return this
            && this.documents
            && this.documents.passport
            && this.documents.passport.filter(passport => passport.revision === this.passport_revision)[0] || {};
    }

    public static getPassportIssueDate() {
        const passport = UserInfoHandler.getPassportData.call(this);

        return passport
            && passport.issue_date
            || EMPTY_DATA;
    }

    public static getPassportIssuePlace() {
        const passport = UserInfoHandler.getPassportData.call(this);

        return passport?.issued_by ?? EMPTY_DATA;
    }

    public static getExpirationDate() {
        const passport = UserInfoHandler.getPassportData.call(this);

        return passport
            && passport.expiration_date
            || EMPTY_DATA;
    }

    public static getDrivingLicenseData() {
        return this
            && this.documents
            && this.documents.driving_license
            && this.documents.driving_license
                .filter(driving_license => driving_license.revision === this.driving_license_revision)[0]
            || {};
    }

    public static getPassportSeries() {
        const passport = UserInfoHandler.getPassportData.call(this);

        return passport.doc_value
            ? `${passport.doc_value.substring(0, PASSPORT_SERIES_FIRST_LIMIT)} `
            + `${passport.doc_value.substring(PASSPORT_SERIES_FIRST_LIMIT, PASSPORT_SERIES_LIMIT)}`
            : EMPTY_DATA;
    }

    public static getPassportNumber() {
        const passport = UserInfoHandler.getPassportData.call(this);

        return passport.doc_value && passport.doc_value.substring(PASSPORT_SERIES_LIMIT);
    }

    public static getPassportReg() {
        const passport = UserInfoHandler.getPassportData.call(this);

        return passport.registration_country || EMPTY_DATA;
    }

    public static getPassportBio() {
        const passport = UserInfoHandler.getPassportData.call(this);

        return passport.biographical_country || EMPTY_DATA;
    }

    public static getPassportGender(): string {
        const passport = UserInfoHandler.getPassportData.call(this);

        return passport.gender || EMPTY_DATA;
    }

    public static getPassportBirthDate(): string {
        const passport = UserInfoHandler.getPassportData.call(this);

        return passport.gender || EMPTY_DATA;
    }

    public static getMechanicTransmissionAllow() {
        return this && this.is_mechanic_transmission_allowed && 'Да' || 'Нет';
    }

    public static getYandexTeamLogin(): string | null {
        const email = this
            && this.setup
            && this.setup.email
            && this.setup.email.address
            && this.setup.email.address.split('@');

        return email && email[1] === 'yandex-team.ru' ? email[0] : null;
    }

    public static getRealtimeData() {
        return this
            && this.realtime_data
            || {};
    }

    public static getUserLink(id: string): string {
        return id && `${location.origin}${location.pathname}#/clients/${id}/info`;
    }
}
