import produce from 'immer';

import { Dict } from '../../types';
import { IStore } from '../components/App/store';
import MY_ROLES_LIST from '../components/Core/classes/MyRoleList';
import {
    BLOCK_RULES_KEY_UI_NEW,
    SETTINGS_KEY_UI,
    SYSTEM_ALERTS,
} from '../components/InterfaceAdminConfig/adminConfigKeys';
import { ISystemAlertItem } from '../components/InterfaceAdminConfig/SystemAlerts/schema';
import { IPostSettingItem, IRule } from '../components/UISettings/component';
import { getCookie } from '../utils/cookie';
import { isValidJSONString } from '../utils/isValidJSONString';

export enum BlockRuleType {
    SHOW = 'show'
}

export type EnthusiasmAction = InitRoles | InitRules | RoleIsLoading | InitPermissions | UpdateAlerts;

export enum INIT {
    INIT_ROLES = 'INIT_ROLES',
    INIT_RULES = 'INIT_RULES',
    ROLE_IS_LOADING = 'ROLE_IS_LOADING',
    INIT_PERMISSIONS = 'INIT_PERMISSIONS',
    UPDATE_ALERTS = 'UPDATE_ALERTS'
}

interface InitRoles {
    type: INIT.INIT_ROLES;
    payload: typeof MY_ROLES_LIST[];
    error?: any;
}

interface InitRules {
    type: INIT.INIT_RULES;
    payload: IPostSettingItem;
    error?: any;
}

interface InitPermissions {
    type: INIT.INIT_PERMISSIONS;
    payload: any;
    error?: any;
}

interface UpdateAlerts {
    type: INIT.UPDATE_ALERTS;
    payload: any;
    error?: any;
}

interface RoleIsLoading {
    type: INIT.ROLE_IS_LOADING;
    payload: boolean;
    error?: any;
}

export function initRoles(roles: typeof MY_ROLES_LIST[], error: any): InitRoles {
    return {
        type: INIT.INIT_ROLES,
        payload: roles,
        error,
    };
}

export function initRules(rules: any, error: any): InitRules {
    return {
        type: INIT.INIT_RULES,
        payload: rules,
        error,
    };
}

export function initPermissions(permissions: any, error: any): InitPermissions {
    return {
        type: INIT.INIT_PERMISSIONS,
        payload: permissions,
        error,
    };
}

export function updateAlerts(alerts: any): UpdateAlerts {
    return {
        type: INIT.UPDATE_ALERTS,
        payload: alerts,
    };
}

export function roleIsLoading(payload: boolean, error: any): RoleIsLoading {
    return {
        type: INIT.ROLE_IS_LOADING,
        payload,
        error,
    };
}

export function getYandexLogin(store: IStore) {
    return store?.AdminUser?.yandex_login;
}

interface IRole {
    roleDescription: string;
    roleDeadLine: number;
    roleGrouppingTags: string;
}

interface IActiveRoles {
    [key: string]: IRole;
}

export interface AdminUserReducerState {
    rolesRaw: MY_ROLES_LIST[];
    activeRoles: IActiveRoles;
    yandex_login: string | undefined;
    userId: string;
    error: any;
    roleIsLoading: boolean;
    rules: IRuleState;
    alerts: ISystemAlertItem[];
    blockRules: Dict<any>;
    rawSettings: any[];
    permissions: any;
}

export interface IRuleState {
    [key: string]: boolean;
}

const initState: AdminUserReducerState = {
    rolesRaw: [],
    activeRoles: {},
    userId: '',
    yandex_login: '',
    roleIsLoading: true,
    error: null,
    blockRules: {},
    rules: {},
    alerts: [],
    rawSettings: [],
    permissions: {},

};

export const CASTOM_RESPONSE_STATUSES = {
    403: '403_CUSTOM',
};

export const EMPTY_ROLES = { response: { status: CASTOM_RESPONSE_STATUSES['403'] } };

const adminUserReducer = (state: AdminUserReducerState = initState, action: EnthusiasmAction) =>
    produce(state, draft => {
        switch (action.type) {
        case INIT.INIT_PERMISSIONS:
            draft.permissions = action.payload;
            break;
        case INIT.UPDATE_ALERTS:
            draft.alerts = action.payload;
            break;
        case INIT.INIT_RULES:
            const setting_rules = action.payload.settings
                    && Array.isArray(action.payload.settings)
                    && action.payload.settings.find(el => el.setting_key === SETTINGS_KEY_UI);

            const alerts_string = action.payload.settings
                    && Array.isArray(action.payload.settings)
                    && action.payload.settings.find(el => el.setting_key === SYSTEM_ALERTS)?.setting_value || '';

            let alerts: ISystemAlertItem[] = [];
            if (isValidJSONString(alerts_string)) {
                alerts = JSON.parse(alerts_string);
            }

            const stringRules = setting_rules && setting_rules.setting_value;
            let _rules: IRule[];
            try {
                _rules = stringRules && JSON.parse(stringRules);
            } catch (e) {
                _rules = [] as IRule[];
            }

            const block_rules = action.payload.settings
                    && Array.isArray(action.payload.settings)
                    && action.payload.settings.find(el => el.setting_key === BLOCK_RULES_KEY_UI_NEW);

            const stringBlockRules = block_rules && block_rules.setting_value;
            let _block_rules: any;
            try {
                _block_rules = stringBlockRules && JSON.parse(stringBlockRules);
            } catch (e) {
                _block_rules = [];
            }

            const blockRules = _block_rules
                ?.reduce((result: Dict<any>, entry: Dict<any>) => {
                    const ruleKey = entry.name;
                    const ruleInfo = entry;
                    let match = false;
                    if (ruleInfo.active) {
                        if (ruleInfo.type === BlockRuleType.SHOW) {
                            const roles = ruleInfo.roles || [];
                            if (roles.length) {
                                match = roles.some(_role => {
                                    _role = _role?.trim();
                                    if (_role && _role.includes('*')) {
                                        const activeRoles = Object.keys(state.activeRoles);
                                        _role = `^${_role.split('*').join('[\\w, -]*')}$`;
                                        const reg = new RegExp(_role, 'ig');

                                        return activeRoles.some(_e => reg.test(_e));
                                    }

                                    if (_role?.[0] === '!') {
                                        const __role = _role?.slice?.(1);

                                        return !state.activeRoles.hasOwnProperty(__role);
                                    }

                                    return state.activeRoles.hasOwnProperty(_role);

                                });
                            }
                        }
                    }

                    result[ruleKey] = match;

                    return result;
                }, {}) || {};

            const rules: IRuleState = _rules && _rules.reduce((_p, _c) => {
                const roles = _c.roles && _c.roles.trim().split(',') || [];
                let match = false;
                if (roles.length) {
                    match = roles.some(_role => {
                        if (_role && _role.includes('*')) {
                            const activeRoles = Object.keys(state.activeRoles);
                            _role = `^${_role.split('*').join('[\\w, -]*')}$`;
                            const reg = new RegExp(_role, 'ig');

                            return activeRoles.some(_e => reg.test(_e));
                        }

                        return state.activeRoles.hasOwnProperty(_role);

                    });
                }

                match && (_p = { ..._p, ...{ [_c.ui]: _c.active } });

                return _p;
            }, {} as IRuleState) || {};

            draft.blockRules = blockRules;
            draft.rules = rules;
            draft.alerts = alerts;
            draft.rawSettings = action.payload.settings || [];
            break;

        case INIT.ROLE_IS_LOADING: {
            draft.roleIsLoading = action.payload;
            break;
        }

        case INIT.INIT_ROLES:
            const activeRoles = Array.isArray(action.payload)
                    && action.payload.reduce((prev: IActiveRoles, cur: typeof MY_ROLES_LIST) => {
                        if (cur.user_role.active === '1' && !prev.hasOwnProperty(cur.role_description.role_id)) {
                            return Object.assign({}, prev, {
                                [cur.role_description.role_id]: {
                                    roleDescription: cur.role_description.role_description,
                                    roleDeadLine: cur.user_role.deadline,
                                    roleGrouppingTags: cur.role_description.role_groupping_tags,
                                },
                            });
                        }

                        return prev;

                    }, {} as IActiveRoles)
                    || {};

            draft.error = action.error || (!Object.keys(activeRoles).length
                ? EMPTY_ROLES
                : null);
            draft.rolesRaw = action.payload;
            draft.yandex_login = getCookie('yandex_login');
            draft.userId = Array.isArray(action.payload)
                    && action.payload[0]
                    && action.payload[0].user_role
                    && action.payload[0].user_role.user_id || '';
            draft.activeRoles = activeRoles;

            return;
        }
    });

export { adminUserReducer, initState };
