import { createSelector } from 'reselect';

import { IStore } from '../components/App/store';
import {
    CALL_STATUS,
    callStatusDescriptions,
    IGetCallIdProps,
    WEBPHONE_STATUS,
    webphoneStatusDescriptions,
} from '../components/Webphone/helpers/webphoneTypes';

export type TCallMetaInfo = {name?: string; id?: string; phoneNumber: string};
type TCallInfo = {client: string; callInfo: IGetCallIdProps};

type TWebphoneMessage = WEBPHONE_STATUS | TCallMetaInfo | TCallInfo | string | number | boolean | null;

export interface IWebphoneReducerState {
    status: WEBPHONE_STATUS;
    callStatus: string | null;
    action: string | null;
    call: TCallInfo | null;
    hold: boolean | null;
    mute: boolean | null;
    callMetaInfo: TCallMetaInfo | null;
    customCallStatusDescription: string | null;
}

export enum WebphoneReducerType {
    SET_STATUS = 'SET_STATUS',
    SET_CALL_STATUS = 'SET_CALL_STATUS',
    REGISTER = 'REGISTER',
    CALL = 'CALL',
    END_CALL = 'END_CALL',
    HOLD_CALL = 'HOLD_CALL',
    MUTE_CALL = 'MUTE_CALL',
    END_ACTION = 'END_ACTION',
    SET_CALL_METAINFO = 'SET_CALL_METAINFO',
    SET_CUSTOM_CALL_DESCRIPTION = 'SET_CUSTOM_CALL_DESCRIPTION'
}

export const WEBPHONE_ACTION_KEYS = {
    [WebphoneReducerType.REGISTER]: 'register',
    [WebphoneReducerType.CALL]: 'call',
    [WebphoneReducerType.HOLD_CALL]: 'hold',
    [WebphoneReducerType.MUTE_CALL]: 'mute',
    [WebphoneReducerType.SET_STATUS]: 'status',
    [WebphoneReducerType.SET_CALL_STATUS]: 'callStatus',
    [WebphoneReducerType.END_CALL]: 'endCall',
    [WebphoneReducerType.SET_CALL_METAINFO]: 'callMetaInfo',
    [WebphoneReducerType.SET_CUSTOM_CALL_DESCRIPTION]: 'customCallStatusDescription',
};

export interface WebphoneAction {
    type: WebphoneReducerType;
    payload?: {
        message: TWebphoneMessage;
    };
}

const webphoneInitialState: IWebphoneReducerState = {
    status: WEBPHONE_STATUS.notRegistered,
    callStatus: null,
    call: null,
    action: null,
    hold: null,
    mute: null,
    callMetaInfo: null,
    customCallStatusDescription: null,
};

export const getWebphone = store => store.Webphone || {};

export const getWebphoneData = createSelector(
    getWebphone,
    webphone => webphone ?? {},
);

export const getWebphoneStatus = createSelector(
    getWebphoneData,
    webphone => webphone?.status ?? null,
);

export const getWebphoneStatusDescription = (store: IStore) => {
    const status = getWebphoneStatus(store);

    return webphoneStatusDescriptions[status] ?? status;
};

export const getCallStatus = createSelector(
    getWebphoneData,
    webphone => webphone?.callStatus ?? null,
);

const getCustomCallStatusDescription = createSelector(
    getWebphoneData,
    webphone => webphone?.customCallStatusDescription ?? null,
);

export const getCallStatusDescription = (store: IStore) => {
    const customCallStatusDescription = getCustomCallStatusDescription(store);

    if (customCallStatusDescription) {
        return customCallStatusDescription;
    }

    const status = getCallStatus(store);

    return callStatusDescriptions[status] ?? status;

};

export const getCallMetaInfo = createSelector(
    getWebphoneData,
    webphone => webphone?.callMetaInfo ?? null,
);

export const isWebphoneReady = (store: IStore) => {
    const status = getWebphoneStatus(store);

    return status === WEBPHONE_STATUS.registered;
};

export const isWebphoneNotRegistered = (store: IStore) => {
    const status = getWebphoneStatus(store);

    return status === WEBPHONE_STATUS.notRegistered
        || status === WEBPHONE_STATUS.registrationFailedDuringCall
        || status === WEBPHONE_STATUS.registrationFailed;
};

export const hasWebphoneActiveCall = (store: IStore) => {
    const callStatus = getCallStatus(store);

    return callStatus !== null && callStatus !== CALL_STATUS.ended && callStatus !== CALL_STATUS.failed;
};

export const hasWebphoneCall = (store: IStore) => {
    const callStatus = getCallStatus(store);

    return callStatus !== null;
};

export const hasWebphoneAcceptedCall = (store: IStore) => {
    const callStatus = getCallStatus(store);

    return callStatus === CALL_STATUS.accepted;
};

export const hasCallEnded = (store: IStore) => {
    const callStatus = getCallStatus(store);

    return callStatus === CALL_STATUS.ended || callStatus === CALL_STATUS.failed;
};

export const setWebphoneStatus = (message: TWebphoneMessage): WebphoneAction => {
    return {
        type: WebphoneReducerType.SET_STATUS,
        payload: {
            message,
        },
    };
};

export const setCallStatus = (message: TWebphoneMessage): WebphoneAction => {
    return {
        type: WebphoneReducerType.SET_CALL_STATUS,
        payload: {
            message,
        },
    };
};

export const registerWebphone = (): WebphoneAction => {
    return {
        type: WebphoneReducerType.REGISTER,
    };
};

export const endCall = (): WebphoneAction => {
    return {
        type: WebphoneReducerType.END_CALL,
    };
};

export const holdCall = (isHoldActive: boolean): WebphoneAction => {
    return {
        type: WebphoneReducerType.HOLD_CALL,
        payload: {
            message: isHoldActive,
        },
    };
};

export const muteCall = (isMuteActive: boolean): WebphoneAction => {
    return {
        type: WebphoneReducerType.MUTE_CALL,
        payload: {
            message: isMuteActive,
        },
    };
};

export const endAction = (): WebphoneAction => {
    return {
        type: WebphoneReducerType.END_ACTION,
    };
};

export const callFromWebphone = (client: string, callInfo: IGetCallIdProps): WebphoneAction => {
    return {
        type: WebphoneReducerType.CALL,
        payload: {
            message: { client, callInfo } ?? null,
        },
    };
};

export const setCallMetaInfo = (metaInfo: TCallMetaInfo | null): WebphoneAction => {
    return {
        type: WebphoneReducerType.SET_CALL_METAINFO,
        payload: {
            message: metaInfo,
        },
    };
};

export const setCustomCallDescription = (message: string | null): WebphoneAction => {
    return {
        type: WebphoneReducerType.SET_CUSTOM_CALL_DESCRIPTION,
        payload: {
            message,
        },
    };
};

export const webphoneReducer = (state: IWebphoneReducerState = webphoneInitialState, action: WebphoneAction) => {
    switch(action.type) {
    case WebphoneReducerType.REGISTER:
        return {
            ...state,
            action: WEBPHONE_ACTION_KEYS[WebphoneReducerType.REGISTER],
        };
    case WebphoneReducerType.END_CALL:
        return {
            ...state,
            action: WEBPHONE_ACTION_KEYS[WebphoneReducerType.END_CALL],
        };
    case WebphoneReducerType.END_ACTION:
        return {
            ...state,
            action: null,
            hold: null,
            call: null,
        };
    case WebphoneReducerType.SET_STATUS:
    case WebphoneReducerType.SET_CALL_STATUS:
    case WebphoneReducerType.SET_CALL_METAINFO:
    case WebphoneReducerType.HOLD_CALL:
    case WebphoneReducerType.MUTE_CALL:
    case WebphoneReducerType.CALL:
    case WebphoneReducerType.SET_CUSTOM_CALL_DESCRIPTION:
        return {
            ...state,
            [WEBPHONE_ACTION_KEYS[action.type]]: action.payload?.message,
        };
    default:
        return state;
    }
};
