import reactPreset from '@bbob/preset-react';
import { render } from '@bbob/react/dist';
import { ReactElement } from 'react';

import { Dict } from '../../../types';

const onlyAllowTags = ['color', 'b', 'i', 'r', 'm', 'l', 'bi', 'mi', 'li', 'line-height', 'url', 'br'];
const DEFAULT_TEXT_SIZE = 16;
const getBbCodeTagSizeAttr = (node: Dict<any>) => {
    return +Object.keys(node?.attrs).filter(attr => !isNaN(+attr))[0] || DEFAULT_TEXT_SIZE;
};

const bbCodePreset = reactPreset.extend((tags) => ({
    ...tags,
    color: node => {
        const color = Object.keys(node?.attrs).filter(attr => attr[0] === '#')[0] ?? '#FFF';

        return {
            tag: 'span',
            attrs: { style: { color } },
            content: node.content,
        };
    },
    b: node => {
        const fontSize = getBbCodeTagSizeAttr(node);

        return {
            tag: 'span',
            attrs: { style: { fontSize, fontWeight: 700 } },
            content: node.content,
        };
    },
    i: node => {
        const fontSize = getBbCodeTagSizeAttr(node);

        return {
            tag: 'span',
            attrs: { style: { fontSize, fontStyle: 'italic' } },
            content: node.content,
        };
    },
    r: node => {
        const fontSize = getBbCodeTagSizeAttr(node);

        return {
            tag: 'span',
            attrs: { style: { fontSize, fontWeight: 400 } },
            content: node.content,
        };
    },
    m: node => {
        const fontSize = getBbCodeTagSizeAttr(node);

        return {
            tag: 'span',
            attrs: { style: { fontSize, fontWeight: 500 } },
            content: node.content,
        };
    },
    l: node => {
        const fontSize = getBbCodeTagSizeAttr(node);

        return {
            tag: 'span',
            attrs: { style: { fontSize, fontWeight: 300 } },
            content: node.content,
        };
    },
    bi: node => {
        const fontSize = getBbCodeTagSizeAttr(node);

        return {
            tag: 'span',
            attrs: { style: { fontSize, fontWeight: 700, fontStyle: 'italic' } },
            content: node.content,
        };
    },
    mi: node => {
        const fontSize = getBbCodeTagSizeAttr(node);

        return {
            tag: 'span',
            attrs: { style: { fontSize, fontWeight: 500, fontStyle: 'italic' } },
            content: node.content,
        };
    },
    li: node => {
        const fontSize = getBbCodeTagSizeAttr(node);

        return {
            tag: 'span',
            attrs: { style: { fontSize, fontWeight: 300, fontStyle: 'italic' } },
            content: node.content,
        };
    },
    ['line-height']: node => {
        const lineHeight = getBbCodeTagSizeAttr(node);

        return {
            tag: 'span',
            attrs: { style: { lineHeight: `${lineHeight}px` } },
            content: node.content,
        };
    },
    url: node => {
        const link = Object.keys(node?.attrs)?.[0] || node.content?.[0];

        return {
            tag: 'a',
            attrs: { href: link, target: '_blank' },
            content: node.content,
        };
    },
    br: node => {
        return {
            tag: 'br',
            content: node.content,
        };
    },
}));

const checkLinks = (text: string | null): string | null => {
    if (text && text.indexOf('http') > -1) {
        const arr = text?.split(' ');

        return arr?.reduce((_p: any, _c: any) => {
            if (_c.indexOf('http') > -1) {
                const temp = _c.split('\n');
                _c = temp.reduce((_p, _c, index) => {
                    if (_c.indexOf('http') === 0) {
                        _p = `${_p}${index !== 0 ? '\n' : ''}[url]${_c}[/url]`;
                    } else {
                        _p = `${_p}${index !== 0 ? '\n' : ''}${_c}`;
                    }

                    return _p;
                }, '');
            }

            return [..._p, _c];
        }, []).join(' ') || null;
    }

    return text;

};

export const getRenderedBBCode = (text: string): ReactElement | null | string => {
    let formattedText = checkLinks(text);
    if (formattedText?.match(/\[\d/)
        || (formattedText?.match(/\[.*\s.*\]/ig) && !formattedText?.match(/\].*\s.*\[/ig)) //check space in tag but not between tags
    ) {
        return formattedText;
    }

    formattedText = formattedText?.replace(/\n/ig, '[br]') || null;
    const isStringValid = formattedText && validateBBTags(formattedText);

    let result: any;
    try {
        if (!isStringValid) {
            result = text;
        } else {
            result = render(formattedText, bbCodePreset(), onlyAllowTags);
        }
    } catch (e) {
        result = text;
    }

    return result;
};

export function validateBBTags(text) {
    const OPEN_BR = '[';
    const CLOSE_BR = ']';

    let openTagsCount = 0;
    const brackets: string[] = [];

    for (let i = 0; i < text.length; i++) {
        if (text[i] === OPEN_BR) {
            brackets.push(OPEN_BR);

            //валидация тега внутри []
            const closeBracketInd = text.indexOf(CLOSE_BR, i);
            if (closeBracketInd !== -1) {
                const tagContent = text.substring(i + 1, closeBracketInd);
                const isClosingTag = tagContent[0] === '/';
                const isValidOpenedTag = tagContent === 'br';

                if (!isClosingTag && !isValidOpenedTag) {
                    openTagsCount++;

                    let stopIteration = false;
                    let ind = onlyAllowTags.length - 1;

                    while (!stopIteration && ind >= 0) {
                        const tag = onlyAllowTags[ind];
                        const isTagAtStart = tagContent.indexOf(tag) === 0;

                        if (tagContent.includes(tag) && isTagAtStart) {
                            const closingTagInd = text.indexOf(`[/${tag}]`);

                            if (closingTagInd === -1) {
                                stopIteration = true;

                            }
                        }

                        ind--;
                    }
                } else {
                    !isValidOpenedTag && openTagsCount--;
                }

            }
        }

        if (text[i] === CLOSE_BR) {
            if (brackets[brackets.length - 1] === OPEN_BR) {
                brackets.length = brackets.length - 1;
            } else {
                brackets.push(CLOSE_BR);
            }
        }
    }

    return !brackets.length && !openTagsCount;
}
