import React from 'react';

import Spin from '../../components/Spin';
import { ytLogs } from '../../utils/sendLogs';
import { buttonLocationDetails, buttonNameDetails } from '../../utils/sendLogs/eventTypes/buttonDetails';
import { EVENT_TYPES } from '../../utils/sendLogs/eventTypes/eventTypes';
import style from './index.css';

export enum ButtonTypes {
    positive = 'positive',
    negative = 'negative',
    warning = 'warning',
}

export interface IButtonProps {
    children: any;
    onClick?: (data?: any, event?: any) => void;
    disabled?: boolean;
    basic?: boolean;
    colorType?: ButtonTypes;
    isLoading?: boolean;
    className?: string;
    file?: boolean;
    fileAccept?: string;
    ytLog?: { button_name: buttonNameDetails; location?: buttonLocationDetails };
    buttonRef?: React.RefObject<HTMLButtonElement>;
    childrenRef?: React.RefObject<HTMLSpanElement>;
}

export class Button extends React.Component<IButtonProps> {
    log: ytLogs | null = null;

    constructor(props: IButtonProps) {
        super(props);

        if (props.ytLog) {
            this.log = ytLogs.getInstance();
        }
    }

    openFile(callback) {
        const file: HTMLInputElement = document.createElement('input');
        file.type = 'file';
        file.style.display = 'none';
        file.multiple = true;
        if (this.props.fileAccept) {
            file.accept = this.props.fileAccept;
        }

        document.body.appendChild(file);
        file.click();

        file.onchange = (e: any) => {
            e.stopPropagation();
            e.preventDefault();
            callback(e.target.files);
            file.remove();
        };
    }

    onClick(event) {
        if (!this.props.disabled && !this.props.isLoading) {
            if (this.props.file) {
                this.openFile((data) => {
                    this.props.onClick && this.props.onClick(data, event);
                });
            } else {
                if (this.props.ytLog && this.log) {
                    this.log.send({
                        data: {
                            event_type: EVENT_TYPES.CLICK_BUTTON,
                            ...this.props.ytLog,
                        },
                    });
                }

                this.props.onClick && this.props.onClick(event);
            }
        }
    }

    render() {
        return <button onClick={this.onClick.bind(this)}
                       ref={this.props.buttonRef}
                       className={`${style.button}` +
                       `${this.props.colorType ? ` ${style[this.props.colorType]}` : ` ${style.primary}`}` +
                       `${this.props.basic ? ` ${style.basic}` : ''}` +
                       `${this.props.isLoading ? ` ${style.loading_button}` : ''}` +
                       `${this.props.disabled ? ` ${style.disabled}` : ''}` +
                       `${this.props.className ? ` ${this.props.className}` : ''}`}>
            <span ref={this.props.childrenRef}
                  className={this.props.isLoading ? style.title_loading : ''}>
                {this.props.children}
            </span>
            {this.props.isLoading
                ? <div className={style.spin_container}>
                    <Spin size={'x'} className={style.spin}/>
                </div>
                : null}
        </button>;
    }
}

const CANCEL_BUTTON_TEXT = 'Отмена';

export interface ICustomButtonProps {
    onClick?: (data?: any, event?: any) => void;
    disabled?: boolean;
    isLoading?: boolean;
    className?: string;
}

export const CancelButton = React.memo((props: ICustomButtonProps) => {
    return <Button basic
                   colorType={ButtonTypes.negative}
                   onClick={props.onClick}
                   disabled={props.disabled}
                   isLoading={props.isLoading}
                   className={props.className}>
        {CANCEL_BUTTON_TEXT}
    </Button>;
});

const DELETE_BUTTON_TEXT = 'Удалить';

export const DeleteButton = React.memo((props: ICustomButtonProps) => {
    return <Button colorType={ButtonTypes.negative}
                   onClick={props.onClick}
                   disabled={props.disabled}
                   isLoading={props.isLoading}
                   className={props.className}>
        {DELETE_BUTTON_TEXT}
    </Button>;
});

const CLEAR_BUTTON_TEXT = 'Очистить';

export const ClearButton = React.memo((props: ICustomButtonProps) => {
    return <Button colorType={ButtonTypes.warning}
                   basic
                   onClick={props.onClick}
                   disabled={props.disabled}
                   isLoading={props.isLoading}
                   className={props.className}>{CLEAR_BUTTON_TEXT}</Button>;
});

const SAVE_BUTTON_TEXT = 'Сохранить';

export const SaveButton = React.memo((props: ICustomButtonProps) => {
    return <Button colorType={ButtonTypes.positive}
                   onClick={props.onClick}
                   disabled={props.disabled}
                   isLoading={props.isLoading}
                   className={props.className}>
        {SAVE_BUTTON_TEXT}
    </Button>;
});
