import * as React from 'react';

import { useDropDirection } from '../../hooks/useDropDirection';
import { useOutsideClickHandler } from '../../hooks/useOutsideClickHandler';
import { Button, ButtonTypes } from '../Button';
import styles from './index.css';

interface IButtonWithPopupProps {
    popupContent: React.ReactNode;
    className?: string;
    colorType?: ButtonTypes;
    displayArrowIcon?: boolean;
    disabled?: boolean;
}

export const ButtonWithPopup = (props: React.PropsWithChildren<IButtonWithPopupProps>) => {
    const { popupContent, colorType, children, className, displayArrowIcon, disabled } = props;
    const [isPopupOpen, setPopup] = React.useState(false);
    const popupRef = React.useRef<HTMLDivElement>(null);
    const buttonRef = React.useRef<HTMLButtonElement>(null);
    const childrenRef = React.useRef<HTMLSpanElement>(null);
    const [verticalDropDirection, horizontalDropDirection] = useDropDirection(popupRef, isPopupOpen);

    useOutsideClickHandler(popupRef, (event) => {
        if (!(event.target.contains(buttonRef?.current) || event.target.contains(childrenRef?.current))) {
            setPopup(false);
        }
    });

    return <div className={`${styles.container} ${
        colorType
            ? styles[`containerColor_${colorType}`]
            : styles.containerColor_primary
    } ` +
        `${styles[`verticalDropDirection_${verticalDropDirection}`]} ` +
        `${styles[`horizontalDropDirection_${horizontalDropDirection}`]} ` +
        `${styles[`dropDirection_${verticalDropDirection}_${horizontalDropDirection}`]}`}>
        {isPopupOpen && (
            <div className={styles.popup}
                 ref={popupRef}>
                {popupContent}
            </div>
        )}
        <Button childrenRef={childrenRef}
                buttonRef={buttonRef}
                disabled={disabled}
                onClick={() => {
                    setPopup((isPopupOpen) => !isPopupOpen);
                }}
                className={`${styles.popupButton} ` +
                `${isPopupOpen ? styles.buttonPopupOpen : ''} ` +
                `${displayArrowIcon ? styles.displayArrowIcon : ''} ` +
                `${className ?? ''}`
                }
                colorType={colorType}
                basic>
            {displayArrowIcon
                ? isPopupOpen
                    ? '▼'
                    : '▲'
                : children}
        </Button>
    </div>;
};
