import 'react-datetime/css/react-datetime.css';

import moment from 'moment';
import React from 'react';
import Datetime from 'react-datetime';
import ReactTooltip from 'react-tooltip';

import { Dict } from '../../../types';
import { ONE_SECOND } from '../../constants';
import { Cross } from '../Cross';
import { UIStatusTypes, UITextComponent } from '../index';
import style from './index.css';

const DATE_FORMAT = 'DD.MM.YYYY';
const TIME_FORMAT = 'HH:mm';
const DESCRIPTION_TOOLTIP_ID = 'DESCRIPTION_TOOLTIP_ID';
const MIN_FREE_SPACE_PIXELS = 300;
const RU_LOCALE = 'ru-ru';
const TIMESTAMP_MIN_LENGTH = 10;
const TIMESTAMP_MAX_LENGTH = 13;

interface IDatePickerProps extends UITextComponent {
    timeFormat?: string;
    onlyDate?: boolean;
    utc?: boolean;
    value: number | string | moment.Moment | Date | null;
    onChange: (value: any) => void;
}

interface IDatePickerState {
    value: number | ''; //string - only '' for empty value
    isInputEmpty: boolean;
    isDateValid: boolean;
    openDown: boolean;
}

export default class DatePicker extends React.Component<IDatePickerProps, IDatePickerState> {
    state: IDatePickerState = {
        value: '',
        isInputEmpty: true,
        isDateValid: true,
        openDown: true,
    };

    constructor(props: IDatePickerProps) {
        super(props);
        const value = this.getValueFromProps();
        this.state = {
            value,
            isInputEmpty: value === '',
            isDateValid: value === '' ? true : moment(value).isValid(),
            openDown: true,
        };
    }

    componentDidUpdate(prevProps: Readonly<IDatePickerProps>): void {
        if (this.props.value !== prevProps.value) {
            if (!this.isValueNaN(this.props.value) && !this.isValueNaN(prevProps.value)) {
                const value = this.getValueFromProps();
                this.setState({
                    value,
                    isInputEmpty: value === '',
                    isDateValid: value === '' ? true : moment(value).isValid(),
                });
            }
        }
    }

    isValueNaN(value: number | string | moment.Moment | Date | null) {
        return typeof value === 'number' ? isNaN(value as number) : false;
    }

    getValueFromProps() {
        const { value: valueProps } = this.props;

        const value: number | '' = valueProps !== undefined
        && valueProps !== null && valueProps !== '' && !this.isValueNaN(valueProps)
            ? +moment(valueProps)
            : '';

        return value;
    }

    onFocus(event: any) {
        const position = event?.target?.getBoundingClientRect();

        const y = position?.y ?? 0;
        const vh = Math.max(document?.documentElement?.clientHeight ?? 0, window?.innerHeight ?? 0);
        const openDown = vh - y > MIN_FREE_SPACE_PIXELS;
        this.setState({ openDown });
    }

    getTimeFormat() {
        return this.props.timeFormat ?? TIME_FORMAT;
    }

    getFullDateFormat() {
        if (!this.props.onlyDate) {
            return `${DATE_FORMAT} ${this.getTimeFormat()}`;
        }

        return DATE_FORMAT;

    }

    onDateTimeValueChange(value: string | moment.Moment) {
        if (value === '') {
            this.onClear();
        } else {
            if (moment(value, this.getFullDateFormat(), true).isValid()) {
                const countedValue = moment(value, this.getFullDateFormat(), true);

                this.setState({ value: +countedValue, isInputEmpty: false, isDateValid: true }, () => {
                    this.props.onChange(+countedValue);
                });
            } else {
                const isNum = Number.isInteger(+value);
                const valueLength = (+value + '').length;
                const isMicroFormat = valueLength === TIMESTAMP_MIN_LENGTH;
                const isMilliFormat = valueLength === TIMESTAMP_MAX_LENGTH;
                const isCorrectLength = isMicroFormat || isMilliFormat;
                const isValid = moment(+value).isValid();

                if (isNum && isCorrectLength && isValid) {
                    const momentValue = isMilliFormat ? moment(+value) : moment(+value * ONE_SECOND);
                    const countedValue = moment(momentValue, this.getFullDateFormat(), true);

                    this.setState({ value: +countedValue, isInputEmpty: false, isDateValid: true }, () => {
                        this.props.onChange(+countedValue);
                    });
                } else {
                    this.setState({ isDateValid: false, isInputEmpty: false });
                }
            }
        }
    }

    onClear() {
        if (!this.props.disabled) {
            this.setState({ value: '', isInputEmpty: true, isDateValid: true }, () => {
                this.props.onChange('');
            });
        }
    }

    render() {
        const {
            placeholder: placeholderProps, status: statusProps, className: classNameProps,
            required, onlyDate, utc, disabled, description,
        } = this.props;
        const { value, isDateValid, openDown, isInputEmpty } = this.state;

        const timeFormat = this.getTimeFormat();
        const status = isDateValid
            ? statusProps
            : {
                type: UIStatusTypes.negative,
                text: this.getFullDateFormat(),
            };
        const placeholder = `${placeholderProps ?? ''}`
            + ` ${status?.text ? ` (${status.text})` : ''}`;

        const inputProps: Dict<any> = {
            placeholder,
            disabled: disabled,
            onFocus: this.onFocus.bind(this),
        };
        if (isInputEmpty) {
            inputProps.value = '';
        }

        return <div className={`${style.date_picker} `
            + ` ${status ? style[`status_${status.type}`] : ''} `
            + `${required ? style.required : ''} `
            + `${classNameProps ?? ''}`}>
            <div className={`${style.placeholder}`
            + ` ${!isInputEmpty ? style.placeholder_is_visible : ''}`}
                 title={placeholder}>
                {placeholder || ''}
            </div>
            {description
                ? <>
                    <div data-tip data-for={DESCRIPTION_TOOLTIP_ID} className={style.description_control}>?</div>
                    <ReactTooltip className={style.description_text}
                                  id={DESCRIPTION_TOOLTIP_ID}
                                  effect='solid'
                                  place='left'>
                        {description}
                    </ReactTooltip>
                </>
                : null}
            <div className={style.controls}>
                <Cross className={disabled ? style.disabled_cross : ''}
                       onClick={this.onClear.bind(this)}/>
            </div>
            <Datetime className={`${style.date_picker} ${!openDown ? style.direction_top : ''}`}
                      dateFormat={DATE_FORMAT}
                      timeFormat={onlyDate
                          ? false
                          : timeFormat}
                      locale={RU_LOCALE}
                      utc={utc}
                      inputProps={inputProps}
                      value={value !== ''
                          ? moment(value)?.format(this.getFullDateFormat())
                          : ''}
                      onChange={this.onDateTimeValueChange.bind(this)}/>
        </div>;
    }
}
