import React from 'react';

import { Dict } from '../../../../types';
import Spin from '../../../components/Spin';
import LS from '../../../utils/localStorage/localStorage';
import { Request2 } from '../../../utils/request';
import { CancelButton, SaveButton } from '../../Button';
import { Window } from '../../FullModal';
import * as coreStyle from '../../index.css';
import { Input } from '../../Input';
import * as style from '../index.css';
import { MDS_REQUESTS, REQUESTS } from '../request';

interface IFileInputModalProps {
    onWindowClose: () => void;
    setPathName: (value: string) => void;
    media: Dict<any> ;
    fileName: string;
    dirName: string;
}

interface IFileInputModalState {
    error: Error | null;
    fileName: string;
    dirName: string;
    pathName: string;
    isPathNameUnique: boolean;
    isLoading: boolean;
}

export default class FileInputModal extends React.Component<IFileInputModalProps, IFileInputModalState> {
    state: IFileInputModalState = {
        error: null,
        fileName: '',
        dirName: '',
        pathName: '',
        isPathNameUnique: true,
        isLoading: false,
    };
    request = new Request2({ requestConfigs: MDS_REQUESTS });

    componentDidMount(): void {
        this.setState({
            dirName: this.props.dirName || '',
            fileName: this.props.fileName || '',
        });
    }

    onNameInputChange(fileName: string) {
        this.setState({ fileName });
    }

    onDirInputChange(dirName: string) {
        this.setState({ dirName });
    }

    onSaveImg() {
        const { fileName, dirName } = this.state;
        const pathName = this.generateFullPathName(fileName, dirName);
        this.checkIfPathNameUnique(pathName);
    }

    generateLocalPathName(fileName: string, dirName: string): string {
        return `${dirName ? encodeURIComponent(dirName) : ''}${dirName ? '/' : ''}${fileName
            ? encodeURIComponent(fileName)
            : ''}`;
    }

    generateFullPathName(fileName: string, dirName: string): string {
        const ls = new LS();
        const prefix = ls.envIsTesting()
            ? `https://s3.mdst.yandex.net/carsharing/`
            : `https://carsharing.s3.yandex.net/`;

        return `${prefix}${this.generateLocalPathName(fileName, dirName)}`;
    }

    checkIfPathNameUnique(pathName: string) {
        this.request.exec(REQUESTS.GET_MDS_RESOURCES)
            .then(response => {
                const mediaArray = response?.keys || [];

                if (mediaArray?.includes(pathName)) {
                    this.setState({
                        isPathNameUnique: false,
                    });
                } else {
                    this.saveImg();
                }
            })
            .catch(error => {
                this.setState({ error });
            });
    }

    saveImg() {
        const { fileName, dirName } = this.state;
        const media = this.props.media;
        const localPath = this.generateLocalPathName(fileName, dirName);
        const fullPath = this.generateFullPathName(fileName, dirName);

        this.setState({ isLoading: true, isPathNameUnique: true }, () => {
            this.request.exec(REQUESTS.ADD_MDS_RESOURCES, {
                queryParams: {
                    key: localPath,
                },
                file: media,
                headers: {
                    'Content-Type': media?.type || 'image',
                },
            })
                .then(() => {
                    this.props.setPathName(fullPath);
                })
                .catch((error) => {
                    this.setState({ pathName: fullPath, error, isLoading: false });
                });
        });
    }

    render() {
        const { onWindowClose } = this.props;
        const { error, dirName, fileName, isLoading, isPathNameUnique } = this.state;

        return <Window onClose={onWindowClose.bind(this)} title={'Загрузка файла'} error={error}>
            {isLoading
                ? <Spin/>
                : <>
                    <Input value={dirName}
                           onChange={this.onDirInputChange.bind(this)}
                           placeholder={'Название директории'}/>

                    <Input value={fileName}
                           required={true}
                           onChange={this.onNameInputChange.bind(this)}
                           placeholder={'Название файла'}/>

                    {!isPathNameUnique
                        ? <div className={style.warning}>Такое имя файла уже существует</div>
                        : null
                    }

                    <div className={coreStyle.button_container}>
                        <CancelButton onClick={onWindowClose.bind(this)}/>
                        <SaveButton onClick={this.onSaveImg.bind(this)} disabled={!Boolean(fileName)}/>
                    </div>
                </>
            }
        </Window>;
    }
}
