import React from 'react';

import UploadIcon from '../../../svg-components/upload.component.svg';
import { Dict } from '../../../types';
import { Request2 } from '../../utils/request';
import { Button } from '../Button';
import { UIStatus, UIStatusTypes } from '../index';
import { Input } from '../Input';
import FileInputModal from './FileInputModal/component';
import * as style from './index.css';
import { MDS_REQUESTS, REQUESTS } from './request';

interface IFileInputProps {
    className?: Dict<any>;
    placeholder?: string;
    value?: string;
    description?: string;
    autoFocus?: boolean;
    onBlur?: (value: any) => void;
    onSubmit?: () => void;
    onChange: (pathName: string) => void;
    dirName?: string;
    required?: boolean;
    disabled?: boolean;
}

interface IFileInputState {
    fileName: string;
    pathName: string;
    mediaFile: Dict<any>;
    isEditWindowOpen: boolean;
    status: UIStatus | null;
}

export const INIT_VALUE_ERROR = 'ERROR';

export default class FileInput extends React.Component<IFileInputProps, IFileInputState> {
    state: IFileInputState = {
        fileName: '',
        pathName: '',
        mediaFile: {},
        isEditWindowOpen: false,
        status: null,
    };

    componentDidMount(): void {
        if (this.props.value) {
            this.setState({
                pathName: this.props.value || '',
            });
        }

        if(this.props.value === INIT_VALUE_ERROR) {
            this.setState({
                status: { type: UIStatusTypes.negative, text: 'Ошибка при загрузке файла' },
            });
            this.setPathName('');
        }
    }

    componentDidUpdate(prevProps: Readonly<IFileInputProps>): void {
        if (this.props.value !== prevProps.value) {
            this.setState({
                pathName: this.props.value || '',
                status: prevProps.value !== INIT_VALUE_ERROR ? null : this.state.status,
            });
        }
    }

    uploadImg(data) {
        const file = data[0] && new Blob([data[0]], { type: data[0]?.type }) || {};
        this.setState({
            isEditWindowOpen: true,
            fileName: data[0]?.name ?? '',
            mediaFile: file,
        });
    }

    onWindowClose() {
        this.setState({
            isEditWindowOpen: false,
            fileName: '',
            mediaFile: {},
        });
    }

    setPathName(pathName: string) {
        if (this.state.isEditWindowOpen) {
            this.setState({ pathName, isEditWindowOpen: false });
        } else {
            this.setState({ pathName });
        }

        this.props.onChange(pathName);
    }

    render() {
        const { pathName, isEditWindowOpen, mediaFile, fileName, status } = this.state;
        const {
            className,
            autoFocus,
            placeholder,
            onBlur,
            onSubmit,
            dirName,
            disabled,
            required,
            description,
        } = this.props;

        const buttonIcon = <UploadIcon className={style.button_icon}/>;

        return <div className={`${style.container} ${className}`}>
            <Input value={pathName}
                   autoFocus={Boolean(autoFocus)}
                   status={status ?? null}
                   className={style.input}
                   onBlur={onBlur && onBlur.bind(this)}
                   onSubmit={onSubmit && onSubmit.bind(this)}
                   onChange={this.setPathName.bind(this)}
                   placeholder={placeholder ?? ''}
                   required={required}
                   description={description}
                   disabled={disabled}/>

            <Button basic
                    file
                    onClick={this.uploadImg.bind(this)}
                    disabled={disabled}
                    className={style.button_upload}>
                {buttonIcon}
            </Button>
            {isEditWindowOpen
                ? <FileInputModal media={mediaFile}
                                  dirName={dirName ?? ''}
                                  fileName={fileName}
                                  onWindowClose={this.onWindowClose.bind(this)}
                                  setPathName={this.setPathName.bind(this)}/>
                : null
            }
        </div>;
    }
}

export class FileInputST extends FileInput {
    request = new Request2({ requestConfigs: MDS_REQUESTS });

    uploadImg(data) {
        const media = data[0] && new Blob([data[0]], { type: data[0]?.type }) || {};

        const contentType = media?.type;
        const docName = data[0]?.name;

        this.setState({ status: null }, () => {
            this.request.exec(REQUESTS.ADD_ST_RESOURCES, {
                file: media,
                queryParams: {
                    file_name: docName,
                },
                headers: {
                    'Content-Type': contentType || 'image',
                },
            })
                .then(response => {
                    this.setPathName(response?.attachment_id ?? '');
                })
                .catch(() => {
                    this.setState({ status: { type: UIStatusTypes.negative, text: 'Ошибка при загрузке файла' } });
                    this.setPathName('');
                });
        });
    }
}
