import React, { useEffect, useState } from 'react';

import CloseIcon from '../../../svg-components/close.component.svg';
import FormatDate from '../FormatDate';
import { FullModal } from '../FullModal';
import { Input } from '../Input';
import style from './index.css';

interface IFileListViewProps {
    files: File[];
    removeItem: (index: number) => void;
    withInput?: boolean;
    inputNameArr?: string[];
    replaceInputNameArr?: (index: number, newValue: string) => void;
}

export class FileListView extends React.Component<IFileListViewProps> {
    render() {
        return <div className={style.file_list}>
            <ul>
                {
                    this.props.files.map((file, index) => {
                        return <FileItem file={file}
                                         key={file.name}
                                         removeItem={this.props.removeItem.bind(this, index)}
                                         withInput={this.props.withInput}
                                         index={index}
                                         inputNameArr={this.props.inputNameArr}
                                         replaceInputNameArr={this.props.replaceInputNameArr}/>;
                    })
                }
            </ul>
        </div>;
    }
}

const FileItem = React.memo((props: {
    file: File;
    removeItem: () => void;
    withInput?: boolean;
    index: number;
    inputNameArr?: string[];
    replaceInputNameArr?: (index: number, newValue: string) => void;
}) => {
    const { file, removeItem, withInput, index, inputNameArr, replaceInputNameArr } = props;
    const [base64, setSrc] = useState<string | undefined>();
    const [isBigger, showBigger] = useState(false);
    const [inputNameValue, setInputNameValue] = useState('');

    const kilobyte = 1024;
    const fractionDigits = 2;

    const getSize = (b) => {
        if (b < kilobyte) {
            return `${(b).toFixed(fractionDigits)}byte`;
        }

        if (b < kilobyte * kilobyte) {
            return `${(b / kilobyte).toFixed(fractionDigits)}Kb`;
        }

        return `${(b / kilobyte / kilobyte).toFixed(fractionDigits)}M`;
    };

    const handleInputNameValueChange = (value) => {
        if(replaceInputNameArr) {
            replaceInputNameArr(index, value);
        }

        setInputNameValue(value);
    };

    useEffect(() => {
        if(inputNameArr) {
            setInputNameValue(inputNameArr[index]);
        }
    },[]);

    const isImg = file?.type?.includes('image');

    const reader = new FileReader();

    reader.readAsDataURL(file);
    reader.onload = () => {
        setSrc(reader.result as string ?? undefined);
    };

    return <li className={style.itemContainer}>
        {
            isBigger && <FullModal onClose={showBigger.bind(null, false)} isMinimal={false}>
                {isImg
                    ? <img src={base64} className={style.img_big}/>
                    : <video src={base64} className={style.video_big} autoPlay controls/>
                }
            </FullModal>
        }
        <span className={style.remove} onClick={removeItem}>
            <CloseIcon/>
        </span>
        <span className={style.prev_container}>
            {isImg
                ? <img src={base64} onClick={showBigger.bind(null, true)}/>
                : <video src={base64} onClick={showBigger.bind(null, true)}/>
            }
        </span>
        <FormatDate value={file.lastModified} withSecond/>
        {
            withInput ?
                <Input value={inputNameValue}
                       dontShowPlaceholderIfFocused
                       onChange={handleInputNameValueChange}
                       className={style.nameInput}
                       placeholder={'Необходимо название файла'}/>
                : <span>{file.name}</span>
        }
        <span>
            <strong>{getSize(file.size)}</strong>
        </span>
    </li>;
});
