import moment from 'moment';
import * as React from 'react';

import { EMPTY_DATA, ONE_DAY, ONE_HOUR, ONE_MINUTE, ONE_SECOND, UTC_OFFSET } from '../../constants';

interface IProps {
    value: any;
    onlyDate?: boolean;
    onlyTime?: boolean;
    withoutYear?: boolean;
    fullTime?: boolean;
    withSecond?: boolean;
    className?: string;
}

export default function FormatDate(props: IProps) {
    moment.locale('ru');
    moment.updateLocale('ru', {
        monthsShort: ['ЯНВ', 'ФЕВ', 'МАР', 'АПР', 'МАЙ', 'ИЮН', 'ИЮЛ', 'АВГ', 'СЕН', 'ОКТ', 'НОЯ', 'ДЕК'],
    });

    return <span className={`formatted-date ${props.className || ''}`}>
        {props.value && moment(props.value).isValid()
            ? moment(props.value).utcOffset(UTC_OFFSET).format(props.onlyDate
                ? 'DD MMM YYYY'
                : props.onlyTime
                    ? props.withSecond
                        ? 'HH:mm:ss'
                        : 'HH:mm'
                    : props.withSecond
                        ? 'DD MMM YYYY, HH:mm:ss'
                        : props.withoutYear
                            ? 'DD MMM, HH:mm'
                            : props.fullTime
                                ? 'DD MMM, HH:mm:ss:ms'
                                : 'DD MMM YYYY, HH:mm',
            ).toUpperCase()
            : '—'}
    </span>;
}

export function FormatDateInString(props: IProps) {
    moment.locale('ru');
    moment.updateLocale('ru', {
        monthsShort: ['ЯНВ', 'ФЕВ', 'МАР', 'АПР', 'МАЙ', 'ИЮН', 'ИЮЛ', 'АВГ', 'СЕН', 'ОКТ', 'НОЯ', 'ДЕК'],
    });

    return props.value && moment(props.value).isValid()
        ? moment(props.value).utcOffset(UTC_OFFSET).format(props.onlyDate
            ? 'DD MMM YYYY'
            : props.onlyTime
                ? 'HH:mm'
                : props.withSecond
                    ? 'DD MMM YYYY, HH:mm:ss'
                    : props.withoutYear
                        ? 'DD MMM, HH:mm'
                        : props.fullTime
                            ? 'DD MMM, HH:mm:ss:ms'
                            : 'DD MMM YYYY, HH:mm',
        ).toUpperCase()
        : '—';
}

export const prettySeconds = (lag: number): string => {
    let rezult = '';
    if (lag < ONE_MINUTE / ONE_SECOND) {
        rezult = lag.toFixed(1) + ' cек.';
    }

    if (lag >= ONE_MINUTE / ONE_SECOND && lag < ONE_HOUR / ONE_SECOND) {
        rezult = (lag / (ONE_MINUTE / ONE_SECOND)).toFixed(1) + ' мин.';
    }

    if (lag >= ONE_HOUR / ONE_SECOND && lag < ONE_DAY / ONE_SECOND) {
        rezult = (lag / (ONE_HOUR / ONE_SECOND)).toFixed(1) + ' час.';
    }

    if (lag >= ONE_DAY / ONE_SECOND) {
        rezult = (lag / (ONE_DAY / ONE_SECOND)).toFixed(1) + ' дн.';
    }

    return rezult;
};

export const dateLag = (date: any, tilDate?: any) => {
    const lag: any = ((tilDate || new Date().getTime()) - date) / ONE_SECOND;

    return prettySeconds(lag) || EMPTY_DATA;
};

export const reverseDate = (data: string) => {
    const tempIndex = 2;
    const temp = data.split('.');

    return temp[1] + '.' + temp[0] + '.' + temp[tempIndex];
};

export const isEqualDate = (pair: any[]) => {
    return Math.abs(new Date(pair[0] * ONE_SECOND).getTime() - new Date(pair[1]).getTime()) < ONE_SECOND;
};

export const timeDurationNow = (ms: number) => {
    const duration = new Date(new Date().toISOString()).getTime() - ms * ONE_SECOND,
        days = Math.floor(duration / ONE_DAY);

    return (days ? (days + 'дн. ') : '') + moment.utc(duration).format('HH:mm:ss');
};

export const displayDuration = (ms: number, withSeconds = false) => {
    const days = Math.floor(ms / ONE_DAY);

    return withSeconds
        ? (days ? (days + 'дн. ') : '') + moment.utc(ms).format('HH:mm:ss')
        : (days ? (days + 'дн. ') : '') + moment.utc(ms).format('HHч mmм');
};

export const durationBetween = (sec: number[]) => {
    const duration = (sec[0] * ONE_SECOND || new Date(new Date().toISOString()).getTime()) - sec[1] * ONE_SECOND;

    return displayDuration(duration, true);
};

export const getAge = (date: string, fromDate?: Date) => {
    const FRACTION_DIGITS = 2;

    let years: any = '—', duration = 0;
    if (date !== '—') {
        duration = (fromDate || new Date()).getTime() - new Date(date).getTime();
        years = (moment.duration(duration, 'milliseconds').asYears() || 0).toFixed(FRACTION_DIGITS);
    }

    return years;
};

export const stringToScheduleTime = (time: string) => {
    //100  --> 01:00
    //1245 --> 12:45
    const HOUR_RANGE = 2;
    const MINUTE_RANGE = -2;
    const TIME_MAX_LENGTH = 4;

    const _time = new Array(TIME_MAX_LENGTH - time.toString().length)
        .fill('0').concat(time.toString().split('')).join('');

    const h = _time.substring(0, HOUR_RANGE);
    const m = _time.substr(MINUTE_RANGE);

    return `${h}:${m}`;
};

export const toUTCDateTime = (timestamp: number) => {
    const UTC_LENGTH = 19;

    return new Date(timestamp).toJSON()?.slice(0, UTC_LENGTH);
};

export const numberToTimeFormat = (number: number) => {
    const ONE_DAY_FULL_HOURS = 23;
    if (number < 0 || number > ONE_DAY_FULL_HOURS || (number !== 0 && !number)) {
        return EMPTY_DATA;
    }

    const MAX_NUM_WITHOUT_NULL = 9;

    return number <= MAX_NUM_WITHOUT_NULL ? `0${number}:00` : `${number}:00`;

};
