import * as React from 'react';
import * as ReactDom from 'react-dom';
import ReactTooltip from 'react-tooltip';

import { SimpleError } from '../../components/SimpleError';
import { isObjectEqual } from '../../utils/isObjectEqual';
import { Button, ButtonTypes } from '../Button';
import { Collapse2 } from '../Collapse2';
import { Cross } from '../Cross';
import style from './index.css';
import { IModal } from './types';

interface IProps {
    title?: string | React.ReactElement;
    onClose: () => void;
    JSONData?: object;
    error?: any;
    collapseError?: any;
    className?: string;
    description?: string | React.ReactElement;
    closeWithConfirm?: boolean;
    noScrollWhenError?: boolean;
    horizontalAutoScroll?: boolean;
    disableMinimizeButton?: boolean;
}

export const fullModalRoot: HTMLElement | null = document.querySelector('#main');

export class FullModal extends React.Component<any, any> {
    el: HTMLElement;

    constructor(props: any) {
        super(props);
        this.el = document.createElement('div');
        this.el.classList.add(style.wrap);
        props.className && this.el.classList.add(props.className);
        this.el.addEventListener('mousedown', this.exit.bind(this));
        this.el.setAttribute('tabIndex', '99');
        this.el.onkeyup = this.key.bind(this);
    }

    key(e: KeyboardEvent) {
        const ESC_CODE = 27;
        if (e.keyCode === ESC_CODE && !this.props.disableCloseModal) {
            this.props.onClose && !this.props.isMinimal && this.props.onClose();
        }
    }

    exit(e: KeyboardEvent) {
        if (e.target === this.el && !this.props.disableCloseModal) {
            this.props.onClose && this.props.onClose();
        }
    }

    componentDidMount() {
        fullModalRoot && fullModalRoot.appendChild(this.el);
        this.el && this.el.focus();
    }

    componentDidUpdate(prevProps: Readonly<any>, prevState: Readonly<any>, snapshot?: any) {
        if (this.props !== prevProps) {
            this.toggleMinimal();
        }
    }

    toggleMinimal() {
        this.el.classList.toggle(style.minimal, this.props.isMinimal);
    }

    componentWillUnmount() {
        fullModalRoot && fullModalRoot.removeChild(this.el);
        this.el.blur();
    }

    render() {
        // @ts-ignore
        return ReactDom.createPortal(
            this.props.children,
            this.el,
        );
    }
}

const DESCRIPTION_ID = 'description-item';

interface IState {
    isCloseConfirmOpen: boolean;
    selectOverflow: boolean;
    isMinimal: boolean;
}

interface IConfirmProps extends IModal {
    question: string | React.ReactElement;
    error: any;
    accept: () => void;
    isWorking?: boolean;
    additionalControls?: any;
    acceptTitle?: string;
    declineTitle?: string;
    ignoreCancelBtn?: boolean;
    disableMinimizeButton?: boolean;
}

export interface IWithConfirmState {
    confirmIsOpen: boolean;
    question: string | React.ReactElement;
    error: any;
    accept: () => void;
    isWorking: boolean;
}

export class Confirm extends React.Component<IConfirmProps> {
    render() {
        const question: any = this.props.question;

        return <Window title={this.props.title || 'Подтвердите действие'}
                       className={this.props.className}
                       disableMinimizeButton={this.props.disableMinimizeButton}
                       error={this.props.error}
                       onClose={!this.props.isWorking && this.props.onClose.bind(this) || undefined}>
            <div className={`${typeof question === 'string' ? style.content : ''}`}>
                {question}
            </div>
            <div className={style.controls}>
                {
                    !this.props.ignoreCancelBtn
                    && <div>
                        <Button disabled={this.props.isWorking}
                                colorType={ButtonTypes.negative}
                                basic
                                onClick={this.props.onClose.bind(this)}>{this.props.declineTitle ?? 'Отмена'}</Button>
                    </div>
                }
                {
                    this.props.additionalControls || null
                }
                <div>
                    <Button isLoading={this.props.isWorking}
                            onClick={this.props.accept}>{this.props.acceptTitle ?? 'Да'}</Button>
                </div>
            </div>
        </Window>;
    }
}

export class Window extends React.Component<IProps, IState> {
    state: IState = {
        isCloseConfirmOpen: false,
        selectOverflow: false,
        isMinimal: false,
    };

    childrenContainerRef = React.createRef<HTMLDivElement>();
    containerRef = React.createRef<HTMLDivElement>();
    observer;

    componentDidMount(): void {
        ReactTooltip.rebuild();

        this.setWindowOverflow();

        window.addEventListener('resize', this.setWindowOverflow.bind(this));

        const mutationConfig = {
            childList: true,
            subtree: true,
        };

        this.observer = new MutationObserver(this.setWindowOverflow.bind(this));
        if (this.containerRef?.current) {
            this.observer.observe(this.containerRef.current, mutationConfig);
        }
    }

    setWindowOverflow() {
        const WINDOW_PADDING = 140;
        if (this.containerRef?.current?.clientHeight
            && window?.innerHeight
            && this.containerRef.current?.clientHeight < window.innerHeight - WINDOW_PADDING) {
            this.setState({
                selectOverflow: true,
            });
        } else if (this.state.selectOverflow) {
            this.setState({
                selectOverflow: false,
            });
        }
    }

    componentDidUpdate(prevProps: Readonly<IProps>) {
        if (this.props.error !== prevProps.error && !isObjectEqual(this.props.error, prevProps.error)
            && !this.props.noScrollWhenError) {
            this.scrollToTop();
        }
    }

    scrollToTop() {
        if (this.childrenContainerRef?.current?.scrollTop) {
            this.childrenContainerRef.current.scrollTop = 0;
        }
    }

    onClose() {
        if (!this.state.isMinimal) {
            if (this.props.closeWithConfirm) {
                this.setState({ isCloseConfirmOpen: true });
            } else {
                this.props.onClose?.();
            }
        }
    }

    closeConfirm() {
        this.setState({ isCloseConfirmOpen: false });
    }

    componentWillUnmount() {
        this.observer.disconnect();
        window.removeEventListener('resize', this.setWindowOverflow);
    }

    minimal() {
        this.setState((prev) => {
            return {
                isMinimal: !prev.isMinimal,
            };
        });
    }

    render() {
        return <FullModal onClose={this.onClose.bind(this)} isMinimal={this.state.isMinimal}>
            <div className={`${style.container} ${this.state.selectOverflow ? style.selectOverflow : ''} `
            + ` ${this.props.horizontalAutoScroll ? style.horizontalAutoScroll : ''} ${this.props.className || ''}`
            }>
                <div className={`${style.title} ${this.props.noScrollWhenError && this.props.error
                    ? style.errorTitle
                    : ''}`}
                     onClick={this.props.noScrollWhenError && this.props.error && this.scrollToTop.bind(this)}>
                    <div>{this.props.title}</div>
                    <div className={style.title__controls}>
                        {this.props.description
                            ? <>
                                <div data-tip data-for={DESCRIPTION_ID} className={style.title__controls_item}>?</div>
                                <ReactTooltip id={DESCRIPTION_ID} effect="solid">
                                    {this.props.description}
                                </ReactTooltip>
                            </>
                            : null}
                        {
                            !this.props.disableMinimizeButton
                            && <div onClick={this.minimal.bind(this)} className={style.minimal_control}>
                                {this.state.isMinimal ? 'ᐃ' : 'ᐁ'}
                            </div>
                        }

                        <div className={`${style.title__controls_item} ${this.state.isMinimal
                            ? style.disabled_cross
                            : ''}`}
                             onClick={this.onClose.bind(this)}>
                            <Cross className={`${style.cross} ${this.state.isMinimal ? style.disabled_cross : ''}`}/>
                        </div>
                    </div>
                </div>
                <div className={`${style.children} ${this.state.selectOverflow ? style.selectChildren : ''} `
                    + ` ${this.props.horizontalAutoScroll ? style.horizontalAutoScrollChildren : ''} `
                    + ` ${this.state.isMinimal ? style.minimal_children : ''}`
                }
                     ref={this.containerRef}>
                    <div>
                        {this.props.error && <SimpleError error={this.props.error}/>}
                        {this.props.collapseError && <Collapse2 expandText={'Посмотреть код ошибки'}
                                                                initialExpanded={false}
                                                                title={`Ошибка`}
                                                                headerClassname={style.collapse_header}>
                            <SimpleError error={this.props.collapseError}
                                         className={style.simpleError}/>
                        </Collapse2>}
                    </div>
                    <div ref={this.containerRef}>
                        {this.props.children}
                    </div>
                </div>
            </div>
            {this.state.isCloseConfirmOpen
                ? <Confirm onClose={this.closeConfirm.bind(this)}
                           error={null}
                           disableMinimizeButton={true}
                           accept={this.props.onClose.bind(this)}
                           question={'Вы действительно хотите закрыть окно?'}/>
                : null}
        </FullModal>;
    }
}
