import React from 'react';

import { GlobalSearchProvider } from '../../../decorators/GlobalSearchProvider';
import { GlobalSearchProviderExport } from '../../../decorators/GlobalSearchProvider/types';
import { SchemaItemVisual } from '../../components/FormConstructor/types';
import { EMPTY_DATA } from '../../constants';
import { UserInfoHandler } from '../../models/user';
import { isObjectEqual } from '../../utils/isObjectEqual';
import Select, { IOptionInfo, SelectDataTypes } from '../Select';
import { GLOBAL_SEARCH_OBJECTS } from './constants';

interface ISuggestingIDSelectProps extends GlobalSearchProviderExport {
    onSelect: (data: any[]) => void;
    placeholder?: string;
    initialValues?: any[];
    multiSelect?: boolean;
    disabled?: boolean;
    required?: boolean;
    className?: string;
    containerClassName?: string;
    dataType?: SelectDataTypes;
    addingOddVariants?: boolean;
    withoutAutoFocus?: boolean;
    object?: GLOBAL_SEARCH_OBJECTS;
    description?: string;
}

interface ISuggestingIDSelectState {
    options: IOptionInfo[];
}

@GlobalSearchProvider()
export default class IDSelect extends React.Component<ISuggestingIDSelectProps, ISuggestingIDSelectState> {
    state: ISuggestingIDSelectState = {
        options: [],
    };

    componentDidUpdate(prevProps: Readonly<ISuggestingIDSelectProps>): void {
        if (!isObjectEqual(this.props, prevProps)) {
            this.generateOptions(this.props.searchData?.objects);
        }
    }

    onSearchValueChange(searchValue) {
        if (searchValue) {
            this.props.search && this.props.search({
                has_all_of: searchValue,
                object: this.props.object || GLOBAL_SEARCH_OBJECTS.all,
            });
        } else {
            this.setState({ options: [] });
        }
    }

    generateOptions(data) {
        const options: IOptionInfo[] = [];

        if (data?.cars?.length) {
            data.cars.forEach(car => {
                const option = {
                    value: car.id || EMPTY_DATA,
                    text: `${car?.number || EMPTY_DATA} | ${car?.model_id || EMPTY_DATA}`,
                    selectedDisplayText: car.id || EMPTY_DATA,
                };

                options.push(option);
            });
        }

        if (data?.users?.length) {
            data.users.forEach(user => {
                const uih = UserInfoHandler;
                const option = {
                    value: user.id || EMPTY_DATA,
                    text: uih.getPrintName.call(user) || EMPTY_DATA,
                    selectedDisplayText: user.id || EMPTY_DATA,
                };

                options.push(option);
            });
        }

        this.setState({ options });
    }

    render() {
        const { options } = this.state;
        const {
            onSelect, placeholder, initialValues, multiSelect, disabled, className, required,
            containerClassName, dataType, addingOddVariants, withoutAutoFocus, searchIsLoading,
            description,
        } = this.props;

        return <Select options={options}
                       required={required}
                       isLoading={searchIsLoading}
                       description={description}
                       onSearchValueChange={this.onSearchValueChange.bind(this)}
                       onSelect={onSelect.bind(this)}
                       placeholder={placeholder || ''}
                       initialValues={initialValues ?? []}
                       multiSelect={multiSelect}
                       disabled={disabled}
                       className={className}
                       containerClassName={containerClassName}
                       dataType={dataType}
                       addingOddVariants={addingOddVariants}
                       withoutAutoFocus={withoutAutoFocus}
                       visual={SchemaItemVisual.ID_SELECT}/>;
    }
}
