import React, { RefObject } from 'react';
import ReactTooltip from 'react-tooltip';

import { Cross } from '../Cross';
import { Window } from '../FullModal';
import { UITextComponent } from '../index';
import { Link } from '../Link';
import style from './index.css';

const LINK_PART = 'https://';
const PRECISION = 10;
const ENTER_KEY = 13;

interface IInput extends UITextComponent {
    value: string | number;
    autoFocus?: boolean;
    onBlur?: (value: any) => void;
    onSubmit?: () => void;
    tabIndex?: number;
    type?: string;
    min?: number;
    max?: number;
    precision?: number;
    hasImage?: boolean;
    imageHandler?: (url: string) => void;
    onKeyDown?: (value: any) => void;
    onClick?: () => void;
    maxLength?: number;
    templateList?: string[] | null;
    focusedClassName?: string;
    dontShowPlaceholderIfFocused?: boolean;
}

interface IInputState {
    focused: boolean;
    value: string | number;
    isTemplateModalOpen: boolean;
}

export class Input extends React.Component<IInput, IInputState> {
    state: IInputState = {
        focused: false,
        value: '',
        isTemplateModalOpen: false,
    };
    inputRef: RefObject<HTMLInputElement> | null = null;

    constructor(props: IInput) {
        super(props);
        this.inputRef = React.createRef();
        this.state = { focused: false, value: this.props.value ?? '', isTemplateModalOpen: false };
    }

    componentDidUpdate(prevProps: Readonly<IInput>, prevState: Readonly<IInputState>, snapshot?: any): void {
        if (this.props.value !== prevProps.value) {
            this.setState({ value: this.props.value });
        }
    }

    onContainerFocus() {
        this.inputRef?.current?.focus();
        this.setState({ focused: true });
    }

    onContainerBlur(e) {
        const { onBlur } = this.props;
        this.setState({ focused: false }, () => {
            if (onBlur) {
                onBlur(e.target.value);
            }
        });
    }

    isValueLink() {
        const { value } = this.props;
        const inputValue: string = value?.toString();
        if (inputValue?.includes(LINK_PART)) {
            return <Link href={inputValue} target={'_blank'}> (ссылка)</Link>;
        }

        return null;

    }

    openTemplatesModal() {
        this.setState({ isTemplateModalOpen: true });
    }

    closeTemplatesModal() {
        this.setState({ isTemplateModalOpen: false });
    }

    onTemplateClick(template: string) {
        this.onChange(template);
        this.closeTemplatesModal();
    }

    onChange(value: string) {
        const { type, onChange } = this.props;
        this.setState({ value }, () => {
            onChange(type === 'number' ? +value : value);
        });
    }

    id = Math.random();

    render() {
        const {
            hasImage, precision, placeholder: placeholderProps, status,
            className, required, imageHandler, tabIndex,
            disabled, min, max, type, onSubmit, onKeyDown, onClick, description, maxLength,
            templateList,
        } = this.props;
        const { focused, isTemplateModalOpen, value } = this.state;

        const hasImg = hasImage && value;
        const step = precision && Math.pow(PRECISION, -precision);
        const placeholder = `${placeholderProps} ${status && status.text ? ` (${status.text})` : ''} `;
        const focusedClassName = this.props.focusedClassName ? this.props.focusedClassName : style.focused;
        const id = `description_input_${this.id}`;

        return <div className={`${style.input} `
        + `${status ? style[`status_${status.type}`] : ''} `
        + `${className || ''} ${hasImg ? style['has-image'] : ' '} ` + `${required ? style.required : ''} `}>
            {!this.props.dontShowPlaceholderIfFocused && <div className={style.placeholder}>
                <div className={`${style.text} ${((value || value + '' === '0') ? style.show : ' ')}`}>
                    {placeholder} {this.isValueLink()}
                </div>

                <div className={style.side_controls}>
                    {description
                        ? <div className={style.description_icon}>
                            <div data-tip data-for={id} className={style.description_icon}>?</div>
                            <ReactTooltip className={style.description_tooltip}
                                          place={'left'}
                                          id={id}
                                          effect="solid">
                                {description}
                            </ReactTooltip>
                        </div>
                        : null}
                </div>
            </div>}
            {
                hasImg && <div className={style['input-has-image']}
                               onClick={imageHandler && imageHandler.bind(null, value)}>📸</div> || ''
            }
            <div className={`${style.input_container} ${focused ? focusedClassName : ''}`}
                 tabIndex={0}
                 onFocus={this.onContainerFocus.bind(this)}>
                <input ref={this.inputRef}
                       className={templateList ? style.with_templates : ''}
                       value={value}
                       onBlur={this.onContainerBlur.bind(this)}
                       tabIndex={tabIndex}
                       placeholder={placeholder}
                       disabled={disabled}
                       maxLength={maxLength}
                       min={min}
                       step={step}
                       max={max}
                       type={type || ''}
                       onKeyUp={(e: any) => {
                           if (e.keyCode === ENTER_KEY) {
                               onSubmit && onSubmit();
                           }
                       }
                       }
                       onKeyDown={onKeyDown && onKeyDown.bind(null)}
                       onChange={(e: any) => {
                           return this.onChange(e.target.value);
                       }}
                       onClick={(e) => {
                           onClick && onClick();
                           e.stopPropagation();
                       }}/>
                <div className={`${style.controls} ${this.props.dontShowPlaceholderIfFocused
                    ? style.controlsFocusedNoPlaceholder
                    : ''}`}>
                    {templateList ? <div className={style.templates_list}>
                        <div onClick={this.openTemplatesModal.bind(this)}
                             className={style.templates_list_label}>
                            T
                        </div>
                    </div> : null}
                    <Cross onClick={!disabled && this.onChange.bind(this, '')}/>
                </div>
            </div>
            {isTemplateModalOpen
                ? <Window title={`Шаблоны для ${placeholderProps}`} onClose={this.closeTemplatesModal.bind(this)}>
                    {templateList?.map(template => {
                        return <div key={template}
                                    className={style.template}
                                    onClick={this.onTemplateClick.bind(this, template)}>
                            {template}
                        </div>;
                    })}
                </Window>
                : null}
        </div>;
    }
}
