import * as React from 'react';

import { buttonLocationDetails, buttonNameDetails } from '../../utils/sendLogs/eventTypes/buttonDetails';
import { Button, IButtonProps } from '../Button';
import { ErrorSign, ErrorsModal } from '../ErrorBar';
import { Confirm } from '../FullModal';
import * as styles from './index.css';

interface IQuickConfirmActionProps {
    question?: string | React.ReactElement;
    button: IButtonProps;
    className?: string;
    accept: () => Promise<any>;
    showConfirm: boolean;
    ytLog?: { button_name: buttonNameDetails; location?: buttonLocationDetails };
    disableSuccess?: boolean;
    callback?: (res: any) => any;
    id?: string;
}

interface IQuickActionButtonState {
    errorModalIsOpen: boolean;
    confirmIsOpen: boolean;
    isWorking: boolean;
    error: Error | null;
    success: boolean;
    multipleError: Record<string, Error | null>;
    multipleSuccess: Record<string, boolean>;

    [x: number]: boolean;
}

export class QuickActionButton extends React.Component<IQuickConfirmActionProps, IQuickActionButtonState> {
    state: IQuickActionButtonState = {
        errorModalIsOpen: false,
        confirmIsOpen: false,
        error: null,
        isWorking: false,
        success: false,
        multipleError: {},
        multipleSuccess: {},
    };

    controlModal(modal, state: boolean) {
        this.setState({
            [modal]: state,
        });
    }

    acceptWithControls() {
        const { accept, disableSuccess, callback, id } = this.props;
        const currError = id ? { [id]: null } : {};
        const currSuccess = id ? { [id]: false } : {};

        this.setState({
            isWorking: true,
            error: null,
            success: false,
            multipleError: { ...this.state.multipleError, ...currError },
            multipleSuccess: { ...this.state.multipleSuccess, ...currSuccess },
            confirmIsOpen: false,
        }, () => {
            accept()
                .then((res) => {
                    callback && callback(res);

                    const multipleSuccess = id ? { [id]: !disableSuccess } : {};

                    this.setState({
                        isWorking: false,
                        success: !disableSuccess,
                        multipleSuccess: { ...this.state.multipleSuccess, ...multipleSuccess },
                    });
                })
                .catch((error) => {
                    const multipleError = id ? { [id]: error } : {};

                    this.setState({
                        error,
                        confirmIsOpen: false,
                        isWorking: false,
                        multipleError: { ...this.state.multipleError, ...multipleError },
                    });
                });
        });
    }

    onClick() {
        const { showConfirm } = this.props;

        if (showConfirm) {
            this.controlModal('confirmIsOpen', true);
        } else {
            this.acceptWithControls();
        }
    }

    render() {
        const {
            errorModalIsOpen, confirmIsOpen, error, success,
            isWorking, multipleError, multipleSuccess,
        } = this.state;
        const { button, question, className, showConfirm, id } = this.props;
        const hasError = (!id && error) || (id && multipleError[id]);
        const isSuccess = (!id && success) || (id && multipleSuccess[id]);

        return <>
            {errorModalIsOpen && (
                <ErrorsModal errors={!id ? [error] : [multipleError[id]]}
                             onClose={this.controlModal.bind(this, 'errorModalIsOpen', false)}/>
            )}
            {confirmIsOpen && showConfirm && question && (
                <Confirm onClose={this.controlModal.bind(this, 'confirmIsOpen', false)}
                         accept={this.acceptWithControls.bind(this)}
                         isWorking={false}
                         question={question}
                         error={null}/>
            )}
            <div className={`${styles.quick_confirm_container} ${className ?? ''}`}>
                <Button className={`${styles.button} ${button?.className
                    ? button.className
                    : ''} ${hasError ? styles.error_showing : ''}`}
                        onClick={this.onClick.bind(this)}
                        colorType={button?.colorType}
                        basic={button?.basic}
                        isLoading={isWorking}
                        disabled={button?.disabled}
                        ytLog={this.props.ytLog}>{button?.children}</Button>
                {hasError && (
                    <ErrorSign onClick={this.controlModal.bind(this, 'errorModalIsOpen', true)}/>
                )}
                {isSuccess &&
                <span className={styles.success_sign}/>
                }
            </div>
        </>;
    }
}
