import React from 'react';

import { CustomSettings } from '../../utils/customSettings';
import { ytLogs } from '../../utils/sendLogs';
import { EVENT_TYPES } from '../../utils/sendLogs/eventTypes/eventTypes';
import { ShortError } from '../ErrorBar';
import * as style from './index.css';
import { UrlEditModal } from './UriEditModal';

export interface IRichEditorControlItem {
    title: React.ReactElement | string;
    handle?: Function;
    isActive?: boolean;
    tooltip?: React.ReactElement | string;
    className?: string;
}

export interface IFastAnswer {
    name: string;
    tags: string[];
    text: string;
}

export interface IRichEditorProps extends IRichEditorHandle {
    initData: string;
    additionalControls?: IRichEditorControlItem[];
    className?: string;
    fastAnswers: IFastAnswer[];
    onChange?: (string) => void;
    secondLineControls?: any[];
    onSendAndHide?: (message: any) => void | null;
    tag_id: any;
    originatorName?: string;
    isClosedByDefault?: boolean;
}

export interface IRichEditorHandle {
    onSendText: (text: string) => void;
}

export interface IRichEditorState {
    error: Error | null;
    text: string;
    disabled: boolean;
    statusText: React.ReactElement | string;
    statusError: Error | null;
    fastAnswersIsOpen: boolean;
    urlModalData: string;
    isClosed: boolean;
}

const FAST_ANSWER_MARK = '--';
const CLOSE_CONTROL = ' ▼';
const OPEN_CONTROL = 'Развернуть ▲';

export class RichEditor extends React.Component<IRichEditorProps, IRichEditorState> {
    customSettings: any;
    textArea: React.RefObject<HTMLTextAreaElement>;
    log = ytLogs.getInstance();
    state: IRichEditorState = {
        error: null,
        text: '',
        disabled: false,
        statusText: '',
        statusError: null,
        fastAnswersIsOpen: false,
        urlModalData: '',
        isClosed: false,
    };
    sendControl: IRichEditorControlItem = {
        title: <strong>Отправить</strong>,
        handle: this.sendText.bind(this),
    };
    sendAndHideControl = this.props.onSendAndHide
        ? {
            title: <span>Отправить+скрыть</span>,
            tooltip: 'Отправить сообщение и скрыть из очереди',
            handle: this.onSendAndHide.bind(this),
        }
        : null;
    addNameControl = this.props.originatorName
        ? { title: <span className={style.name_button}>Имя</span>, handle: this.addName.bind(this) }
        : null;
    commonControls: IRichEditorControlItem[] = [
        { title: <strong>B</strong>, handle: this.wrapBB.bind(this, 'b') },
        { title: <span>U</span>, handle: this.wrapBB.bind(this, 'u') },
        { title: <span>I</span>, handle: this.wrapBB.bind(this, 'i') },
        { title: <span>url</span>, handle: this.openUrlModal.bind(this) },
        this.addNameControl as IRichEditorControlItem,
    ];

    constructor(props) {
        super(props);
        this.textArea = React.createRef();
        this.customSettings = new CustomSettings().getSettings();
    }

    componentDidMount(): void {
        this.textArea.current && this.textArea.current.focus();
        this.setInitData();
        this.checkIfClosed();
    }

    componentDidUpdate(prevProps: Readonly<IRichEditorProps>, prevState: Readonly<IRichEditorState>) {
        if (this.props.initData !== prevProps.initData) {
            this.setInitData();
        }

        if (this.props.tag_id !== prevProps.tag_id) {
            this.checkIfClosed();
        }

        if (prevProps.isClosedByDefault === undefined && this.props.isClosedByDefault) {
            this.setState({ isClosed: this.props.isClosedByDefault });
        }
    }

    setInitData() {
        this.props.initData && this.setState({
            text: this.props.initData,
            fastAnswersIsOpen: this.props.initData && (this.props.initData.indexOf(FAST_ANSWER_MARK) !== -1) || false,
        });
    }

    checkIfClosed() {
        const isClosedByProps = Boolean(this.props.isClosedByDefault);

        if (isClosedByProps !== this.state.isClosed) {
            this.closeRichEditor(isClosedByProps);
        }
    }

    sendText(key?: string) {
        typeof key === 'string'
            ? this.logAction(`sendBy${key}Enter`)
            : this.logAction('sendByButton');

        this.props.onSendText(this.state.text);
    }

    onSendAndHide() {
        this.props.onSendAndHide?.(this.state.text);
    }

    wrapBB(tag) {
        const node = this.textArea && this.textArea.current && this.textArea.current;
        const selectedText = document.getSelection();
        node && node.setRangeText(`[${tag}]${selectedText}[/${tag}]`);

        this.logAction(tag);
        this.setNodeValue(node);
    }

    addName() {
        const node = this.textArea && this.textArea.current && this.textArea.current;
        if (node) {
            const finishCursorPosition = node.selectionStart + (this.props.originatorName?.length || 0);

            node.setRangeText(this.props.originatorName ?? '');
            this.setNodeValue(node);

            node.focus();
            node.selectionStart = finishCursorPosition;
        }
    }

    openUrlModal(event, data) {
        const urlModalData = data !== undefined
            ? data
            : document.getSelection()?.toString();
        this.setState({ urlModalData });
    }

    setUrl(url) {
        const node = this.textArea?.current;
        node?.setRangeText(url);
        this.setNodeValue(node);
    }

    setNodeValue(node) {
        node && this.setState({
            text: node.value,
        }, () => {
            this.props.onChange && this.props.onChange(this.state.text);
        });
    }

    logAction(action_name) {
        this.log.send({
            data: {
                event_type: EVENT_TYPES.CLICK_BUTTON,
                button_name: `richEditor_${action_name}`,
            },
        });
    }

    onChange(e: React.ChangeEvent<HTMLTextAreaElement>) {
        this.setText(e.target.value);
    }

    setText(text: string, callback?: any) {
        this.setState({
            text,
        }, () => {
            this.props.onChange && this.props.onChange(this.state.text);

            this.setState({
                fastAnswersIsOpen: this.state.text.indexOf(FAST_ANSWER_MARK) !== -1,
            }, () => {
                callback?.();
            });
        });
    }

    disable(disabled: boolean) {
        this.setState({
            disabled,
        });
    }

    setStatusText(statusText, options?: any) {
        if (options && options.error) {
            this.setState({
                statusText: '',
                statusError: options.errorData,
            });
        } else {
            this.setState({
                statusText,
                statusError: null,
            });
        }
    }

    onKeyUp(e: KeyboardEvent) {
        const isHotkeysWorks = this.customSettings.useHotkeys;
        const hasSendCombination = e.key === 'Enter' && (e.altKey || e.shiftKey || e.ctrlKey);

        if (isHotkeysWorks && hasSendCombination) {
            const key = e.altKey ? 'Alt' : e.shiftKey ? 'Shift' : e.ctrlKey ? 'Ctrl' : '';
            this.sendText(key);
        }
    }

    fastAnswerSelect(fastAnswer) {
        const text = this.state.text;
        const startOfFastAnswer = text.indexOf(FAST_ANSWER_MARK);
        const finalText = text.substr(0, startOfFastAnswer) + fastAnswer;

        this.logAction('fastAnswer');
        this.setText(finalText, () => {
            const cur = this.textArea?.current;
            cur?.focus();
            cur?.setSelectionRange(0, 0);
            cur?.scrollTo({ top: 0 });
        });
    }

    fastAnswerFilter() {
        const text = this.state.text;

        return text.slice(text.indexOf(FAST_ANSWER_MARK) + FAST_ANSWER_MARK.length);
    }

    closeRichEditor(isClosed) {
        this.setState({ isClosed });
    }

    render() {
        const { isClosed } = this.state;

        return <div className={`${style.component}`
        + ` ${this.state.disabled ? style.disabled : ''}`
        + ` ${this.props.className || ''}`}>
            <div className={`${style.head} ${isClosed ? style.head_closed : ''}`}>
                {!this.state.isClosed
                    ? <RichEditorControls disabled={this.state.disabled
                    || !this.state.text
                    || Boolean(this.props.secondLineControls?.length)}
                                          controls={
                                              [
                                                  ...this.props.additionalControls || [],
                                                  ...this.commonControls,
                                                  this.sendControl,
                                                  this.sendAndHideControl as IRichEditorControlItem,
                                              ]}/>
                    : null
                }
                <div className={style.controls_item}
                     id={'openBtn'}
                     title={isClosed ? 'Развернуть' : 'Свернуть'}
                     onClick={this.closeRichEditor.bind(this, !this.state.isClosed)}>
                    {isClosed ? OPEN_CONTROL : CLOSE_CONTROL}
                </div>
            </div>
            {!isClosed
                ? !this.props.secondLineControls?.length
                    ? <>
                        <div className={style.status}>
                            <i>статус:</i> <Status text={this.state.statusText} error={this.state.statusError}/>
                        </div>
                    </>
                    : <RichEditorControls disabled={false}
                                          controls={this.props.secondLineControls}/>
                : null}
            {!isClosed && this.state.fastAnswersIsOpen
                ? <FastAnswers answers={this.props.fastAnswers}
                               filter={this.fastAnswerFilter()}
                               onSelect={this.fastAnswerSelect.bind(this)}/>
                : null
            }
            {!isClosed
                ? <div className={`${style.edit_box}`}>
                    <textarea autoFocus
                              ref={this.textArea}
                              value={this.state.text}
                              onKeyUp={this.onKeyUp.bind(this)}
                              onChange={this.onChange.bind(this)}
                              disabled={this.state.disabled}/>
                </div>
                : null
            }
            {this.state.urlModalData
                ? <UrlEditModal startValue={this.state.urlModalData}
                                closeModal={this.openUrlModal.bind(this, '')}
                                onSaveValue={this.setUrl.bind(this)}/>
                : null
            }
        </div>;
    }
}

const FastAnswers = React.memo((props: { answers: IFastAnswer[]; onSelect: (text) => void; filter: string }) => {
    const filter = props.filter?.toUpperCase()?.trim();
    const arr = filter
        ? props.answers?.filter((item) => {
            return item.name.toUpperCase().includes(filter)
            || item.tags?.some(i => i.toUpperCase().includes(filter))
            || item.text.toUpperCase().includes(filter);
        }) ?? []
        : props.answers ?? [];

    return <div className={style.fast_answers}>
        {!arr?.length
            ? <h3>нет совпадений</h3>
            : arr?.map((item, index) => {
                return <div className={style.fast_answer_item}
                            key={index}
                            onClick={props.onSelect.bind(null, item.text)}>
                    <div className={style.fast_answer_item_header}>
                        <strong>{item.name}</strong> [{item.tags?.join(', ') ?? ''}]
                    </div>
                    <div className={style.fast_answer_item_content}>{item.text}</div>
                </div>;
            })
        }
    </div>;
});

interface RichEditorControls {
    controls: IRichEditorControlItem[];
    disabled: boolean;
}

const RichEditorControls = React.memo((props: RichEditorControls) => {
    return <div className={`${style.controls}`}>
        {props.controls.map((control, index) => {
            return control
                ? <RichEditorControlsItem key={index}
                                          index={index}
                                          control={control}
                                          disabled={control.isActive ? false : props.disabled}/>
                : null;
        })}</div>;
});

const RichEditorControlsItem = React.memo((props: any) => {
    const { index, control, disabled } = props;

    return <div key={index}
                title={control.tooltip}
                className={`${style.controls_item} ${disabled ? style.disabled : ''}`}
                onClick={!disabled ? control.handle
                    && control.handle.bind(null) : null}>{control.title}</div>;
});

interface IStatusProps {
    text: React.ReactElement | string;
    error: Error | null;
}

const Status = React.memo((props: IStatusProps) => {
    return <>
        {
            props.error
                ? <ShortError error={props.error}/>
                : props.text || null
        }
    </>;
});
