import * as React from 'react';

import { Dict } from '../../../../types';
import { UIStatusTypes } from '../../index';
import { Link } from '../../Link';
import Select, { IOptionInfo, ISelectProps } from '../index';

export interface IConstantsFaceTag {
    constants: (traits: string[], force?: boolean) => Promise<Dict<string[]>>;
}

interface ISelectConstantsProps extends IConstantsFaceTag, Omit<ISelectProps, 'options'> {
    reference: string;
}

interface ISelectConstantsState {
    isLoading: boolean;
    loadingError: Error | null;
    options: IOptionInfo[];
}

export default class SelectConstants extends React.Component<ISelectConstantsProps, ISelectConstantsState> {
    state: ISelectConstantsState = {
        isLoading: false,
        loadingError: null,
        options: [],
    };

    componentDidMount(): void {
        this.loadOptions();
    }

    componentDidUpdate(prevProps: Readonly<ISelectConstantsProps>) {
        if (this.props.reference !== prevProps.reference) {
            this.loadOptions();
        }
    }

    loadOptions() {
        const { reference = '' } = this.props;

        this.setState({ isLoading: true, loadingError: null }, () => {
            this.props.constants([reference])
                .then((response => {

                    const optionsTextArray = response?.[reference] ?? [];

                    const options = optionsTextArray.map(optionsText => {
                        return { value: optionsText, text: optionsText };
                    });

                    this.setState({ isLoading: false, options });
                }))
                .catch(loadingError => {
                    this.setState({ loadingError, isLoading: false });
                });
        });
    }

    render() {
        const { options, isLoading, loadingError } = this.state;
        const { placeholder: placeholderProps = '', status: statusProps } = this.props;
        const placeholder = <span>
            {placeholderProps}
            (<Link onClick={this.loadOptions.bind(this)}>Обновить</Link>)
        </span>;

        const status = loadingError
            ? {
                type: UIStatusTypes.negative,
                text: 'Ошибка при загрузке вариантов',
            }
            : statusProps;

        return <div>
            <Select {...this.props}
                    placeholder={placeholder}
                    isLoading={isLoading}
                    disabled={isLoading || !!loadingError}
                    options={options}
                    status={status}/>
        </div>;
    }
}
