import React, { ReactElement } from 'react';
import VirtualList from 'react-tiny-virtual-list';
import ReactTooltip from 'react-tooltip';

import { Dict } from '../../../types';
import { SchemaItemVisual } from '../../components/FormConstructor/types';
import Spin from '../../components/Spin';
import { EMPTY_DATA } from '../../constants';
import { deepCopy } from '../../utils/utils';
import { Cross } from '../Cross';
import { UIComponent } from '../index';
import style from './index.css';

export const WARNING_LABEL = '⚠️';
//Max Options list length in pixels
const MAX_DROP_DOWN = 195;
export const DISPLAY_TEXT_KEY = '__display_text';
const VALUE_IS_OBJECT_ERROR = 'Ошибка! Значение является объектом';

export enum SelectDataTypes {
    string = 'string',
    numeric = 'numeric',
}

export interface IOptionInfo {
    readonly value: any;
    text?: string | ReactElement;
    description?: string | ReactElement | ReactElement[];
    selectedDisplayText?: string;
}

export interface IOption {
    info: IOptionInfo;
    settings: {
        isSelected?: boolean;
        isOdd?: boolean;
    };
}

export interface ISelectProps extends UIComponent {
    options: IOptionInfo[];
    initialValues?: any[] | null;
    multiSelect?: boolean;
    onSelect: (values: any[]) => void;
    onSearchValueChange?: (data: string) => void;
    disabled?: boolean;
    className?: string;
    containerClassName?: string;
    dataType?: SelectDataTypes;
    addingOddVariants?: boolean;
    withoutAutoFocus?: boolean;
    isLoading?: boolean;
    visual?: SchemaItemVisual;
    sortWhenSearching?: (options: IOption[]) => IOption[];
}

interface ISelectState {
    isOptionsListShow: boolean;
    options: IOption[];
    searchValue: string;
    isFocused: boolean;
}

export default class Select extends React.Component<ISelectProps, ISelectState> {
    state: ISelectState = {
        isOptionsListShow: false,
        options: [],
        searchValue: '',
        isFocused: false,
    };
    ignoreBlur = false;
    id = Math.random();

    setIgnoreBlur(): void {
        this.ignoreBlur = true;
    }

    clearIgnoreBlur(): void {
        this.ignoreBlur = false;
    }

    componentDidMount(): void {
        this.formatOptionsData();
    }

    isOptionsNotEqual(optionsProps: IOptionInfo[], optionsPrev: IOptionInfo[]) {
        const options = optionsProps?.map(optionProps => deepCopy(optionProps));

        return options?.length !== optionsPrev?.length
            || options?.some((option, index) => {
                if (option && optionsPrev?.[index]) {
                    return option?.value !== optionsPrev?.[index]?.value;
                }

                return true;
            });
    }

    componentDidUpdate(prevProps: Readonly<ISelectProps>, prevState: Readonly<ISelectState>, snapshot?: any): void {
        const { options, initialValues } = this.props;
        const { options: optionsPrev, initialValues: initialValuesPrev } = prevProps;

        const isOptionNotEqual = this.isOptionsNotEqual(options, optionsPrev);

        const isInitialValuesNotEqual = initialValues && Array.isArray(initialValues)
            ? (initialValues.length !== initialValuesPrev?.length)
            || initialValues.some((initialValue, index) => {
                if (initialValue !== null && initialValue !== undefined
                    && initialValuesPrev?.[index] !== null && initialValuesPrev?.[index] !== undefined) {
                    return initialValue !== initialValuesPrev?.[index];
                }

                return true;
            })
            : initialValues !== initialValuesPrev;

        if (isOptionNotEqual || isInitialValuesNotEqual) {
            this.formatOptionsData();
        }
    }

    handleBlur(): void {
        if (this.ignoreBlur) {
            return;
        }

        this.setState({
            isOptionsListShow: false,
            isFocused: false,
        });
    }

    handleFocus() {
        this.setState({ isFocused: true });
    }

    formatOptionsData(): void {
        const { options: propsOptions = [], initialValues } = this.props;
        const optionsCopied = propsOptions.map(option => deepCopy(option));

        const options: IOption[] = optionsCopied.map(optionCopied => {
            const info = deepCopy(optionCopied);

            if (info && !info.selectedDisplayText) {
                info.selectedDisplayText = info.text !== null && info.text !== undefined
                    ? ['string', 'number', 'bigint', 'boolean'].includes(typeof info.text)
                        ? info.text !== null && info.text !== undefined ? info.text.toString() : EMPTY_DATA
                        : info?.[DISPLAY_TEXT_KEY]
                            ? info[DISPLAY_TEXT_KEY]
                            : Select.collectElementText(info.text)
                    : info.value || EMPTY_DATA;
            }

            return {
                info,
                settings: {
                    isSelected: false,
                    isOdd: false,
                },
            };
        });

        this.setState({ options }, () => {
            initialValues && this.formatInitialData();
        });
    }

    formatInitialData(): void {
        let { initialValues = [], multiSelect } = this.props;
        const { options = [] } = this.state;

        if (!Array.isArray(initialValues)) {
            initialValues = [initialValues];
        }

        if (!multiSelect && initialValues.length > 1) {
            console.warn('More than ONE initialValue is pushed in NOT multiSelect Select');
            initialValues = [initialValues[initialValues.length - 1]];
        }

        initialValues.forEach((value: string) => {
            const isOdd = !options.some(option => {
                if (option.info.value === value) {
                    option.settings.isSelected = true;

                    return true;
                }

                return false;
            });
            if (isOdd) {
                if (typeof value === 'object') {
                    if (Object.keys(value)?.length > 0) {
                        options.push({
                            info: {
                                text: value,
                                value: value,
                            },
                            settings: { isOdd: true, isSelected: true },
                        });
                    }
                } else {
                    options.push({
                        info: {
                            text: value,
                            value: value,
                        },
                        settings: { isOdd: true, isSelected: true },
                    });
                }
            }
        });

        this.setState({ options });
    }

    onSelectClickHandler(): void {
        const { disabled } = this.props;

        this.setIgnoreBlur();
        if (!disabled) {
            this.setState({ isOptionsListShow: true });
        }
    }

    onArrowClick(): void {
        const { disabled } = this.props;
        const { isOptionsListShow } = this.state;

        if (!disabled) {
            this.setIgnoreBlur();
            this.setState({ isOptionsListShow: !isOptionsListShow });
        }
    }

    static collectElementText(element: Dict<any>, hashedString?: string): string {
        let text = hashedString || '';

        const children = element?.props?.children;

        if (typeof children === 'object') {
            if (Array.isArray(children)) {
                children.forEach(child => {
                    if (typeof child === 'string') {
                        text += `${hashedString ? ' ' : ''}${child}`;
                    } else {
                        text += `${hashedString ? ' ' : ''}${this.collectElementText(child)}`;
                    }
                });
            } else {
                text += `${hashedString ? ' ' : ''}${this.collectElementText(children)}`;
            }
        } else {
            text += `${hashedString ? ' ' : ''}${children}`;
        }

        return text;
    }

    onSelect() {
        let selectedValues = this.state.options
            .filter(option => option.settings.isSelected)
            .map(option => option.info.value) ?? [];

        if (!this.props.multiSelect) {
            selectedValues = selectedValues[0] ?? '';
        }

        this.props.onSelect && this.props.onSelect(selectedValues);
    }

    onClearClick() {
        if (this.props.disabled) {
            return;
        }

        const { options } = this.state;
        options.forEach(option => {
            option.settings.isSelected = false;
        });

        this.setState({ options, isOptionsListShow: false }, this.onSelect);
    }

    selectOption(pickedOption: IOption): void {
        const { multiSelect } = this.props;
        const { options: optionsState } = this.state;
        const options = optionsState.map(option => deepCopy(option));
        const isSelected = !!pickedOption.settings.isSelected;
        let isOptionsListShow = false;

        options.forEach(option => {
            if (!multiSelect) {
                option.settings.isSelected = option.info.value === pickedOption.info.value
                    ? !isSelected
                    : false;
            } else {
                if (option.info.value === pickedOption.info.value) {
                    option.settings.isSelected = !isSelected;
                }
            }
        });

        if (multiSelect) {
            isOptionsListShow = true;
        }

        this.setState({ options, isOptionsListShow }, this.onSelect);
    }

    isSelectedOptions(): boolean {
        return this.state.options.some(option => !!option.settings.isSelected);
    }

    updateOptions(options: IOption[]) {
        this.setState({ options }, () => {
            this.onSelect();
        });
    }

    closeOptionList() {
        this.setState({ isOptionsListShow: false });
    }

    render() {
        const {
            multiSelect,
            addingOddVariants,
            withoutAutoFocus,
            isLoading,
            visual,
            description,
            placeholder,
        } = this.props;
        const { options, isFocused, isOptionsListShow } = this.state;

        const selectedCount = options.filter(option => option.settings.isSelected)?.length ?? 0;

        const status = this.props.status;
        const id = `description_select_${this.id}`;

        return <div className={`${style.select_container} `
        + `${status ? style[`status_${status.type}`] : ''} `
        + `${this.props.containerClassName ? this.props.containerClassName : ''} `}>

            <div className={style.placeholder_container}>
                <div className={style.placeholder}>
                    <div>
                        {placeholder ?? ''}
                        {status?.text ? <span title={status?.text ?? ''}> ({status.text})</span> : ''}
                    </div>
                </div>
                {description
                    ? <div>
                        <div data-tip data-for={id} className={style.description_icon}>?</div>
                        <ReactTooltip className={style.description_tooltip}
                                      place={'left'}
                                      id={id}
                                      effect="solid">
                            {description}
                        </ReactTooltip>
                    </div>
                    : null}
            </div>

            <div className={`${style.select} ${this.props.disabled ? style.disabled : ''} `
            + `${isFocused ? style.focused_select : ''} `
            + `${this.props.required ? style.required : ''} `
            + `${this.props.className ? this.props.className : ''} `}
                 onMouseDown={this.setIgnoreBlur.bind(this)}
                 onMouseUp={this.clearIgnoreBlur.bind(this)}
                 onMouseOut={this.clearIgnoreBlur.bind(this)}
                 tabIndex={0}
                 onFocus={this.handleFocus.bind(this)}
                 onBlur={this.handleBlur.bind(this)}>

                <div className={`values_container ${style.values_container}`}
                     onClick={this.onSelectClickHandler.bind(this)}>
                    {isLoading
                        ? <span className={style.span}> <Spin size={'s'}/> </span>
                        : null
                    }

                    {options && Array.isArray(options) && options.length
                        ? options.map(option => {
                            let displayText = option.info?.selectedDisplayText ?? option.info.value;
                            displayText = typeof displayText === 'object' && Object.keys(displayText)?.length === 0
                                ? VALUE_IS_OBJECT_ERROR
                                : displayText;

                            return option.settings.isSelected
                                ? <div title={displayText}
                                       key={option.info.value}
                                       data-odd={option.settings.isOdd}
                                       className={`current_value ${style.current_value} ${option.settings.isOdd
                                           ? style.odd
                                           : ''}`}>
                                    <span>{displayText}</span>
                                </div>
                                : null;
                        })
                        : null}
                </div>
                <div className={style.selected_count} onClick={this.onSelectClickHandler.bind(this)}>
                    {multiSelect
                        ? <div className={style.selected_count_label}>
                            <div title={'Множественный выбор'}>
                                {selectedCount || 'M'}
                            </div>
                        </div>
                        : null}
                </div>
                <div className={style.controls}>
                    <div className={`${style.arrow} ${isOptionsListShow ? style.opened : ''}`}
                         onClick={this.onArrowClick.bind(this)}/>
                    <Cross onClick={this.onClearClick.bind(this)}/>
                </div>
                {isOptionsListShow
                    ? <OptionsList options={options}
                                   sortWhenSearching={this.props.sortWhenSearching?.bind(this)}
                                   withoutAutoFocus={withoutAutoFocus ?? false}
                                   multiSelect={multiSelect ?? false}
                                   updateOptions={this.updateOptions.bind(this)}
                                   addingOddVariants={addingOddVariants ?? false}
                                   onClose={this.closeOptionList.bind(this)}
                                   selectOption={this.selectOption.bind(this)}
                                   onSearchValueChange={this.props.onSearchValueChange?.bind(this)}
                                   isLoading={Boolean(isLoading)}
                                   visual={visual}/>
                    : null}
            </div>
        </div>;
    }
}

interface IOptionsListProps {
    options: IOption[];
    withoutAutoFocus: boolean;
    multiSelect: boolean;
    updateOptions: (options: IOption[]) => void;
    addingOddVariants: boolean;
    onClose: () => void;
    selectOption: (option: IOption) => void;
    isLoading: boolean;
    dataType?: SelectDataTypes;
    onSearchValueChange?: (data: string) => void;
    visual?: SchemaItemVisual;
    sortWhenSearching?: (options: IOption[], searchValue: string) => IOption[];
}

interface IOptionsListState {
    searchValue: string;
    openDown: boolean;
    focusedOptionIndex: null | number;
    rowHeights: number[];
}

const DEFAULT_SIZE = 50;

class OptionsList extends React.Component<IOptionsListProps, IOptionsListState> {
    state: IOptionsListState = {
        searchValue: '',
        openDown: true,
        focusedOptionIndex: null,
        rowHeights: [],
    };
    optionsList: React.RefObject<HTMLDivElement> | null = null;
    input: React.RefObject<HTMLInputElement> | null = null;

    constructor(props: IOptionsListProps) {
        super(props);
        this.optionsList = React.createRef();
        this.input = React.createRef();
    }

    componentDidMount(): void {
        const { withoutAutoFocus } = this.props;
        const position = this.optionsList?.current?.getBoundingClientRect();
        const y = position?.y ?? 0;
        const vh = Math.max(document.documentElement.clientHeight ?? 0, window.innerHeight ?? 0);
        const openDown = vh - y > MAX_DROP_DOWN;

        if (!withoutAutoFocus) {
            this.input?.current?.focus();
        }

        this.setState({ openDown });
    }

    getSnapshotBeforeUpdate(
        prevProps: Readonly<IOptionsListProps>,
        prevState: Readonly<IOptionsListState>,
    ): any | null {
        const selectedOptions = this.props.options?.filter(option => option.settings.isSelected) || [];

        if (selectedOptions?.length && this.optionsList?.current?.scrollHeight) {
            return { scrollPosition: this.optionsList?.current?.scrollHeight - this.optionsList?.current?.scrollTop };
        }

        return null;

    }

    componentDidUpdate(
        prevProps: Readonly<IOptionsListProps>,
        prevState: Readonly<IOptionsListState>,
        snapshot?: any,
    ): void {
        if (snapshot && this.optionsList) {
            const { scrollPosition } = snapshot;

            if (this.optionsList.current) {
                this.optionsList.current.scrollTop = this.optionsList.current.scrollHeight - scrollPosition;
            }
        }
    }

    filterItems(): IOption[] {
        if (this.props.visual !== SchemaItemVisual.ID_SELECT) {
            let options: IOption[] = this.props.options.map(option => deepCopy(option));
            const { searchValue = '' } = this.state;

            if (searchValue) {
                options = options
                    .filter(option => {
                        return Object.values(option.info).some(value => {
                            if (typeof value !== 'string' && typeof value !== 'number') {
                                value = Select.collectElementText(value);
                            }

                            return value?.toString()?.toLowerCase()?.includes(searchValue?.toLowerCase());
                        });
                    },
                    );
            }

            options = options.sort((option1, option2) => {
                const optionSortIndex1 = option1.settings.isSelected ? 1 : 0;
                const optionSortIndex2 = option2.settings.isSelected ? 1 : 0;

                return optionSortIndex2 - optionSortIndex1;
            });

            if (this.props.sortWhenSearching) {
                return this.props.sortWhenSearching(options, searchValue);
            }

            return options;
        }

        return this.props.options;
    }

    onValueChange(e: React.ChangeEvent<HTMLInputElement>): void {
        const NEW_VARIANTS_SPLITTER = ',';
        const { options = [], multiSelect, addingOddVariants = false, dataType } = this.props;

        const selectedValues: IOption[] = options
            .filter(option => option?.settings?.isSelected);

        const searchValue = e?.target?.value ?? '';

        if (addingOddVariants && searchValue
            && searchValue?.toString()?.includes(NEW_VARIANTS_SPLITTER)
            && (multiSelect || !selectedValues.length)) {

            let oddValues: string[] | number[] = searchValue?.toString()?.split(NEW_VARIANTS_SPLITTER)
                ?.filter(value => value);

            if (dataType === SelectDataTypes.numeric) {
                oddValues = oddValues?.map(value => +value);
            }

            this.addNotExistingOption(oddValues);
        } else {
            this.setState({ searchValue });
        }

        this.props.onSearchValueChange && this.props.onSearchValueChange(searchValue);
    }

    addNotExistingOption(values: any[]) {
        const { options = [], updateOptions } = this.props;
        values = values.filter((value: string | number) => value);

        values.forEach((value: string | number) => {
            const isOdd = !options.some(option => {
                if (option?.info?.value === value) {
                    if (option?.settings?.isSelected) {
                        option.settings.isSelected = true;
                    }

                    return true;
                }

                return false;
            });
            if (isOdd) {
                //NOTE: revert RP https://github.yandex-team.ru/carsharing/ya-drive-admin/pull/788
                /*options.forEach(option => {
                    option.settings.isSelected = false;
                });*/
                options.push({
                    info: {
                        text: value.toString(),
                        value: value,
                    },
                    settings: { isOdd: true, isSelected: true },
                });
            }
        });

        this.setState({ searchValue: '' }, () => {
            updateOptions(options);
        });
    }

    onKeyDown(event: KeyboardEvent): void {
        const { selectOption, onClose } = this.props;
        const { key } = event;
        const filteredOptions = this.filterItems();
        const optionsLength = filteredOptions?.length ?? 0;
        const currentFocusedOptionIndex = this.state.focusedOptionIndex;
        let focusedOptionIndex: number | null = null;

        switch (key) {
        case 'ArrowDown':
            if (currentFocusedOptionIndex === null) {
                focusedOptionIndex = 0;
            } else {
                if (currentFocusedOptionIndex + 1 < optionsLength) {
                    focusedOptionIndex = currentFocusedOptionIndex + 1;
                } else {
                    focusedOptionIndex = 0;
                }
            }

            break;
        case 'ArrowUp':
            if (currentFocusedOptionIndex === null) {
                focusedOptionIndex = optionsLength - 1;
            } else {
                if (currentFocusedOptionIndex > 0) {
                    focusedOptionIndex = currentFocusedOptionIndex - 1;
                } else {
                    focusedOptionIndex = optionsLength - 1;
                }
            }

            break;
        case 'Enter':
            currentFocusedOptionIndex !== null && selectOption(filteredOptions[currentFocusedOptionIndex]);
            break;
        case 'Escape':
            this.setState({
                focusedOptionIndex: null,
            }, () => {
                onClose();
            });
            break;
        default:
            focusedOptionIndex = null;
            break;
        }

        this.setState({ focusedOptionIndex });
    }

    setRowHeights(height, index) {
        const newRowHeights: number[] = this.state.rowHeights;

        newRowHeights[index] = height;
        this.setState({ rowHeights: newRowHeights });
    }

    render() {
        const { selectOption, addingOddVariants, isLoading } = this.props;
        const { searchValue, openDown, focusedOptionIndex, rowHeights } = this.state;
        const filteredOptions = this.filterItems();

        const sumHeight = rowHeights.slice(0, filteredOptions.length).reduce((p, c) => p + c, 0);

        const height = Math.min(sumHeight, MAX_DROP_DOWN);

        return <div ref={this.optionsList}
                    className={`${style.options} ${openDown ? style.open_down : style.open_up}`}
                    onKeyDown={this.onKeyDown.bind(this)}>
            <div className={style.search_input_container}>
                <input ref={this.input}
                       value={searchValue}
                       className={style.search_input}
                       placeholder={'Поиск...'}
                       onChange={this.onValueChange.bind(this)}/>
                {addingOddVariants
                    ? <div className={style.ico_add_container}>
                        <Cross className={style.ico_add}
                               onClick={this.addNotExistingOption.bind(this, [searchValue])}/>
                    </div>
                    : null}
                <div className={style.ico_mglass_container}>
                    <div className={style.ico_mglass}/>
                </div>
            </div>

            {filteredOptions?.length
                ? <VirtualList key={filteredOptions.length}
                               width={'100%'}
                               height={height}
                               itemCount={filteredOptions.length}
                               itemSize={(index) => {
                                   return rowHeights[index] ?? DEFAULT_SIZE;
                               }}
                               renderItem={({ index, style }) => {
                                   const item = filteredOptions[index];

                                   return <OptionsItem option={item}
                                                       _style={style}
                                                       key={index}
                                                       index={index}
                                                       setHeight={this.setRowHeights.bind(this)}
                                                       openDown={openDown}
                                                       focusedOptionIndex={focusedOptionIndex}
                                                       selectOption={selectOption.bind(this)}/>;
                               }}/>
                : !isLoading
                    ? <div key={'not-found'}
                           className={`${style.option_text}`}>
                        <div className={style.title}>Нет вариантов</div>
                    </div>
                    : null
            }
        </div>;
    }
}

interface IOptionsItemProps {
    option: IOption;
    index: number;
    focusedOptionIndex: number | null;
    openDown: boolean;
    selectOption: () => void;
    setHeight: (height: number, index: number) => void;
    _style: Dict<any>;
}

interface IOptionsItemState {
}

class OptionsItem extends React.Component<IOptionsItemProps, IOptionsItemState> {
    state: IOptionsItemState = {};
    entityNameRef = React.createRef<HTMLDivElement>();

    componentDidMount(): void {
        const height = this.entityNameRef?.current?.clientHeight;

        if (height) {
            this.props.setHeight(height, this.props.index);
        }
    }

    render() {
        const { option, index, focusedOptionIndex, openDown, selectOption } = this.props;

        const isSelected = option?.settings?.isSelected ?? false;

        let titleDisplayText: string = ['string', 'number', 'bigint', 'boolean'].includes(typeof option.info.text)
            ? (option.info.text?.toString() || option.info.value?.toString())
            : option.info.text ?? option.info.value;
        titleDisplayText = typeof titleDisplayText === 'object' && Object.keys(titleDisplayText)?.length === 0
            ? VALUE_IS_OBJECT_ERROR
            : titleDisplayText;
        let descriptionDisplayText: string = option.info.description !== undefined
            ? option.info.description : option.info.value;
        descriptionDisplayText = typeof descriptionDisplayText === 'object' && descriptionDisplayText !== null
        && Object.keys(descriptionDisplayText)?.length === 0
            ? ''
            : descriptionDisplayText;

        return <div style={Object.assign({}, this.props._style, { width: '100%' })}
                    className={`${style.option} `
                    + `${focusedOptionIndex === index ? style.focused : ''} `
                    + `${option.settings.isOdd ? style.odd : ''} `
                    + `${isSelected ? style.selected : ''} `
                    + `${openDown ? style.opened_down : style.opened_up}`}
                    title={titleDisplayText}
                    onClick={selectOption.bind(this, deepCopy(option))}>
            <div className={style.option_text} ref={this.entityNameRef}>
                <div className={style.title}>
                    {titleDisplayText}
                </div>
                <div className={style.description}>
                    {descriptionDisplayText}
                </div>
            </div>
        </div>;
    }
}
