import React, { RefObject } from 'react';
import ReactTooltip from 'react-tooltip';

import { isValidJSONString } from '../../utils/isValidJSONString';
import { Cross } from '../Cross';
import { UITextComponent } from '../index';
import style from './index.css';

interface ITextareaProps extends UITextComponent {
    json?: boolean;
    withoutCross?: boolean;
    hidePlaceholder?: boolean;
}

interface ITextareaState {
    value: string;
    hasScrollBar: boolean;
}

export default class TextArea extends React.Component<ITextareaProps, ITextareaState> {
    state: ITextareaState = {
        value: this.props.value,
        hasScrollBar: false,
    };
    ref: RefObject<HTMLTextAreaElement>;

    constructor(props: ITextareaProps) {
        super(props);
        this.ref = React.createRef();
    }

    componentDidMount(): void {
        if (this.hasScrollbar()) {
            this.setState({ hasScrollBar: true });
        }
    }

    shouldComponentUpdate(nextProps: Readonly<ITextareaProps>, nextState: Readonly<ITextareaState>): boolean {
        const { value: stateValue, hasScrollBar } = this.state;
        const { value: propsValue, placeholder, disabled, json } = this.props;

        return (stateValue !== nextState.value || propsValue !== nextProps.value ||
            placeholder !== nextProps.placeholder ||
            disabled !== nextProps.disabled ||
            hasScrollBar !== nextState.hasScrollBar ||
            json !== nextProps.json);
    }

    componentDidUpdate(prevProps: Readonly<ITextareaProps>, prevState: Readonly<ITextareaState>): void {
        const { value } = this.props;
        if (value !== prevProps.value) {
            this.setState({ value, hasScrollBar: this.hasScrollbar() });
        }
    }

    onChange(e: React.ChangeEvent<HTMLTextAreaElement>) {
        const { json, onChange } = this.props;
        this.setState(
            { value: e.target.value, hasScrollBar: this.hasScrollbar() }, () => {
                const { value } = this.state;
                if (json) {
                    if (isValidJSONString(value)) {
                        onChange && onChange(value);
                    }

                    if (value === '') {
                        onChange && onChange(null);
                    }
                } else {
                    onChange && onChange(value);
                }
            },
        );
    }

    focus() {
        this.ref && this.ref.current && this.ref.current.focus();
    }

    onClearClick() {
        const { onChange } = this.props;

        this.setState({ value: '' }, () => {
            const { value } = this.state;

            onChange?.(value);
        });
    }

    onMouseUp() {
        this.setState({ hasScrollBar: this.hasScrollbar() });
    }

    hasScrollbar(): boolean {
        const current = this.ref?.current as HTMLTextAreaElement;
        if (current.clientHeight !== undefined && current.scrollHeight !== undefined) {
            const { clientHeight, scrollHeight } = current;

            return clientHeight < scrollHeight;
        }

        return false;
    }

    id = Math.random();

    render() {
        const {
            placeholder: placeholderProps,
            status,
            json,
            className,
            required,
            disabled,
            description,
            hidePlaceholder,
            withoutCross,
        } = this.props;
        const { value, hasScrollBar } = this.state;
        const id = `description_textarea_${this.id}`;

        const placeholder = `${placeholderProps}`
            + `${status?.text ? ` (${status.text})` : ''}`
            + `${json && (value && !isValidJSONString(value)) ? ` - Невалидный JSON` : ``}`;

        return (
            <div className={style.textarea + ' ' + (className || '')
            + ` ${status ? style[`status_${status.type}`] : ''} `
            + ` ${required ? style.required : ''} `
            + ` ${json && (value && !isValidJSONString(value)) ? style.status_negative : ''} `}>
                <div className={style.placeholder_container}>
                    {!hidePlaceholder
                    && <>
                        <div className={style.placeholder + (value && ' ' + style['placeholder_is-visible'])}>
                            {placeholder}
                        </div>
                        {description
                            ? <div>
                                <div data-tip data-for={id} className={style.description_icon}>?</div>
                                <ReactTooltip className={style.description_tooltip}
                                              place={'left'}
                                              id={id}
                                              effect="solid">
                                    {description}
                                </ReactTooltip>
                            </div>
                            : null}
                    </>}

                </div>
                <textarea className={style.textarea_input}
                          value={value}
                          disabled={disabled}
                          placeholder={placeholder}
                          ref={this.ref}
                          tabIndex={0}
                          onMouseUp={this.onMouseUp.bind(this)}
                          onChange={this.onChange.bind(this)}/>
                {!withoutCross &&
                <div className={`${style.options}`
                    + ` ${hasScrollBar ? style.with_scrollbar : ''}`
                    + ` ${hidePlaceholder ? style.without_placeholder: ''}`}>
                    <Cross onClick={this.onClearClick.bind(this)}/>
                </div> || null
                }
            </div>
        );
    }
}
