import React, { ComponentClass, FunctionComponent, RefObject } from 'react';
import VirtualList from 'react-tiny-virtual-list';

import { Dict } from '../../../types';
import { Link } from '../Link';
import * as style from './index.css';

const LIST_ITEM_SIZE = 45;

export interface IPlaneTreeItem<T> {
    data: T;
    meta: {
        [key: string]: any;

        id: string;
        children: IPlaneTreeItem<T>[] | null;
        nestingLevel: number;
        childrenCount?: number;
        generalIndex?: number;
        filteredChildrenCount?: number | null;
        parentsIds?: string[];
        className?: string;
        active?: boolean;
    };
}

type customListDataType = FunctionComponent<any> | ComponentClass<any> | string

interface IVirtualTreeListProps<T> {
    treeBuilder: () => IPlaneTreeItem<T>[];
    treeFilter: () => ((node: IPlaneTreeItem<T>) => boolean) | null;
    isFilterValueExist: boolean;
    treeListItem: customListDataType | [customListDataType, any];
    initCollapsedLevels?: number[]; //init tree with collapsed levels in this array
    showPropositions?: boolean;
}

interface IVirtualTreeListState<T> {
    planeTree: IPlaneTreeItem<T>[];
    collapsedIds: Dict<boolean>;
    activeIds: Dict<boolean>;
    rowHeights: number[];
}

export class VirtualTreeList<T> extends React.Component<IVirtualTreeListProps<T>, IVirtualTreeListState<T>> {
    state: IVirtualTreeListState<T> = {
        planeTree: [],
        collapsedIds: {},
        activeIds: {},
        rowHeights: [],
    };
    listRef: RefObject<HTMLDivElement> | null = null;

    constructor(props: IVirtualTreeListProps<T>) {
        super(props);
        this.listRef = React.createRef();
    }

    componentDidMount(): void {
        this.getPlaneTree();
    }

    componentDidUpdate(prevProps: IVirtualTreeListProps<T>): void {
        const { isFilterValueExist, showPropositions } = this.props;
        const { isFilterValueExist: prevIsFilterValueExist, showPropositions: prevShowPropositions } = prevProps;

        const isFilterCleared = prevIsFilterValueExist && !isFilterValueExist;
        const isFilterAdded = !prevIsFilterValueExist && isFilterValueExist;

        if (isFilterCleared || showPropositions !== prevShowPropositions) {
            this.getPlaneTree();
        }

        if (isFilterAdded) {
            this.unCollapseAll();
        }
    }

    getPlaneTree(recalculateCollapse = true): void {
        const collapsedIds = Object.assign({}, this.state.collapsedIds);
        const activeIds = Object.assign({}, this.state.activeIds);
        const { treeBuilder, initCollapsedLevels } = this.props;
        const tree = treeBuilder?.();
        const planeTree = this.makeTreePlane({ children: tree });

        recalculateCollapse && planeTree.forEach(planeTreeItem => {
            const { meta } = planeTreeItem;
            const { id, active } = meta;
            const nestingLevel = meta.nestingLevel as number;
            activeIds[id] = active ?? false;
            if (initCollapsedLevels?.includes?.(nestingLevel)) {
                collapsedIds[id] = true;
            }
        });

        this.setState({ planeTree, collapsedIds, activeIds });
    }

    makeTreePlane<T>(params: {
        children: IPlaneTreeItem<T>[];
        nestingLevel?: number;
        parentsIds?: string[];
        startIndex?: number;
    }): IPlaneTreeItem<T>[] {
        const { children, nestingLevel = 0, parentsIds = [], startIndex = 0 } = params;

        const getNodeChildrenCount = (node: IPlaneTreeItem<T>, full: boolean) => {
            const { children } = node.meta;
            const firstLevelCount = children
                ? full ? children.length : children?.filter(child => child.meta.active).length
                : 0;
            const nextLevelsCount = children ? children.reduce((result: number, child) => {
                result += getNodeChildrenCount(child, full);

                return result;
            }, 0) : 0;

            return firstLevelCount + nextLevelsCount;
        };

        let indexWithChildren = startIndex;

        return children.reduce((result: IPlaneTreeItem<T>[], child, index) => {
            child.meta.nestingLevel = nestingLevel;
            child.meta.parentsIds = parentsIds;
            child.meta.childrenCount = getNodeChildrenCount(child, false);
            child.meta.generalIndex = index + indexWithChildren;
            indexWithChildren += getNodeChildrenCount(child, true);
            result.push(child);

            if (child.meta.children) {
                result.push(...this.makeTreePlane({
                    children: child.meta.children,
                    nestingLevel: nestingLevel + 1,
                    parentsIds: [...parentsIds, child.meta.id],
                    startIndex: child.meta.generalIndex + 1,
                }));
            }

            return result;
        }, []);
    }

    getListHeight() {
        const windowSize = { width: window.innerWidth, height: window.innerHeight };
        const topPosition = this.listRef?.current?.offsetTop || 0;
        const BOTTOM_MARGIN = 35;

        return +Math.floor(windowSize.height - topPosition - BOTTOM_MARGIN);
    }

    onItemClick(e, item_id: string) {
        const collapsedIds = Object.assign({}, this.state.collapsedIds);
        collapsedIds[item_id] = !collapsedIds[item_id];

        if (e.target?.type !== 'checkbox') {
            this.setState({ collapsedIds });
        }
    }

    getAllNodesChildrenActiveIds(node: IPlaneTreeItem<T>): string[] {
        const { activeIds } = this.state;
        const { children = null } = node.meta;
        const result: string[] = [];

        if (children) {
            result.push(...children.map(child => child.meta.id).filter(childId => activeIds[childId]));
            children.forEach(child => {
                const childIds = this.getAllNodesChildrenActiveIds(child);
                result.push(...childIds);
            });

            return result;
        }

        return result;

    }

    getFilteredTree() {
        const { activeIds } = this.state;
        const { treeFilter: propsTreeFilter, showPropositions } = this.props;
        let planeTree: IPlaneTreeItem<T>[] = this.state.planeTree;
        const treeFilter = propsTreeFilter();

        //Array of filtered items
        let nodesFilteredId: string[] = [];
        const nodesChildrenCount: Dict<string[]> = {};

        //Filtering every tree item with filter from props
        if (treeFilter && !showPropositions) {
            planeTree.forEach(node => {
                if (treeFilter?.(node)) {
                    const childrenIds = this.getAllNodesChildrenActiveIds(node);

                    //count children for current item
                    if (nodesChildrenCount?.[node.meta.id]) {
                        nodesChildrenCount[node.meta.id].push(...childrenIds);
                    } else {
                        nodesChildrenCount[node.meta.id] = [...childrenIds];
                    }

                    nodesChildrenCount[node.meta.id] = Array.from(new Set(nodesChildrenCount[node.meta.id]));

                    //count children for current item's parents
                    node.meta.parentsIds?.forEach((parentsId, index) => {

                        const elementsChildren = [...childrenIds];
                        if (node.meta.active) {
                            elementsChildren.push(node.meta.id);
                        }

                        elementsChildren.push(...(
                            node.meta.parentsIds?.slice(index + 1)?.filter(parentsId => activeIds[parentsId]) ?? []
                        ));
                        if (nodesChildrenCount?.[parentsId]) {
                            nodesChildrenCount[parentsId].push(...elementsChildren);
                        } else {
                            nodesChildrenCount[parentsId] = elementsChildren;
                        }

                        nodesChildrenCount[parentsId] = Array.from(new Set(nodesChildrenCount[parentsId]));
                    });

                    nodesFilteredId
                    && nodesFilteredId.push(node.meta.id, ...(node.meta.parentsIds ?? []), ...childrenIds);
                }
            });
            nodesFilteredId = Array.from(new Set(nodesFilteredId));
        }

        planeTree = planeTree.reduce((result: IPlaneTreeItem<T>[], node: IPlaneTreeItem<T>) => {
            const isCollapsed = node.meta.parentsIds?.some(parentsId => this.state.collapsedIds[parentsId]);
            let isItemFiltered = true;

            if (treeFilter && !showPropositions) {
                isItemFiltered = nodesFilteredId.includes(node.meta.id);
            }

            if (isItemFiltered && !isCollapsed) {
                node.meta.filteredChildrenCount = treeFilter && !showPropositions
                    ? nodesChildrenCount?.[node.meta.id]?.length ?? 0
                    : null;
                result.push(node);
            }

            return result;
        }, []);

        return planeTree;
    }

    unCollapseAll() {
        this.setState({ collapsedIds: {} });
    }

    collapseAll() {
        const collapsedIds = this.state.planeTree?.reduce((result: Dict<boolean>, planeTreeItem) => {
            result[planeTreeItem.meta.id] = true;

            return result;
        }, {});
        this.setState({ collapsedIds });
    }

    setRowHeights(height, index) {
        const newRowHeights: number[] = this.state.rowHeights;

        newRowHeights[index] = height;
        this.setState({ rowHeights: newRowHeights });
    }

    render() {
        const { rowHeights, collapsedIds } = this.state;
        const { showPropositions, treeListItem } = this.props;
        const listHeight = this.getListHeight();
        const planeTree = this.getFilteredTree();

        return <div>
            <div className={style.collapse_controls}>
                <Link onClick={this.unCollapseAll.bind(this)} className={style.collapse_control}>
                    Разложить все
                </Link>
                <Link onClick={this.collapseAll.bind(this)} className={style.collapse_control}>
                    Сложить все
                </Link>
            </div>
            <div ref={this.listRef}>
                {planeTree.length === 0
                    ? <h4>Не найдено элементов дерева</h4>
                    : null}
                <VirtualList height={listHeight}
                             itemCount={planeTree.length}
                             itemSize={(index) => {
                                 return rowHeights[planeTree[index].meta.generalIndex ?? 0] ?? LIST_ITEM_SIZE;
                             }}
                             renderItem={({ index, style }) => {
                                 return <TreeListItem key={`${planeTree[index].meta.id}_${index}`}
                                                      customListData={treeListItem}
                                                      _style={style}
                                                      index={index}
                                                      setHeight={this.setRowHeights.bind(this)}
                                                      onItemClick={this.onItemClick.bind(this)}
                                                      collapsedIds={collapsedIds}
                                                      item={planeTree[index]}
                                                      showPropositions={showPropositions}/>;
                             }}/>
            </div>
        </div>;
    }
}

const TREE_ITEM_MARGIN = 2;
const TREE_ITEM_EXTRA_BORDER_MARGIN = 0.1;

interface ITreeListItemProps<T> {
    customListData: customListDataType | [customListDataType, any];
    _style: Dict<any>;
    index: number;
    onItemClick: () => void;
    collapsedIds: Dict<boolean>;
    item: IPlaneTreeItem<T>;
    setHeight: (height: number, index: number) => void;
    showPropositions?: boolean;
}

class TreeListItem<T> extends React.Component<ITreeListItemProps<T>, {}> {

    entityNameRef = React.createRef<HTMLDivElement>();

    componentDidMount(): void {
        const { item } = this.props;
        const height = this.entityNameRef?.current?.clientHeight;

        if (height) {
            this.props.setHeight(height, item.meta.generalIndex ?? 0);
        }
    }

    render() {
        const { item, _style, collapsedIds, showPropositions } = this.props;
        const { meta } = item || {};
        const { id = '', children = [], nestingLevel = 0 } = meta || {};

        const itemStyle = Object.assign({}, _style,
            {
                marginLeft: `${nestingLevel * TREE_ITEM_MARGIN + TREE_ITEM_EXTRA_BORDER_MARGIN}em`,
                width: `auto`,
                minWidth: `calc(100% - ${nestingLevel * TREE_ITEM_MARGIN + TREE_ITEM_EXTRA_BORDER_MARGIN}em)`,
            });

        return <div style={itemStyle}
                    onClick={(e) => this.props.onItemClick.call(this, e, id)}
                    className={style.tree_node_container}>
            <div className={`${style.tree_node} ${meta.className || ''}`} ref={this.entityNameRef}>
                {children?.length
                    ? <div className={`${style.node_arrow} ${collapsedIds?.[id] ? style.opened : ''}`}>▶</div>
                    : <div/>}
                <div className={style.content}>
                    {
                        React.createElement(this.props.customListData?.[0]
                            || this.props.customListData, {
                            item,
                            showPropositions,
                            ...this.props?.customListData?.[1],
                        })
                    }
                </div>
            </div>
        </div>;
    }
}
