import * as React from 'react';

import { Copy } from '../../components/Copy';
import {
    AccountType,
    CardLevels,
    FRAME_TRANSLATE,
    IAccount,
    ITrustCard,
    IYandexAccount,
    PaymentSystem,
} from '../../components/Settings/Wallets/types';
import { EMPTY_DATA, ONE_SECOND } from '../../constants';
import { rubs } from '../../utils/rubs';
import FormatDate from '../FormatDate';
import { Link } from '../Link';
import * as styles from './index.css';
import { PaymentSystemLogos } from './PaymentSystemLogos';

interface IWalletProps {
    account?: IAccount;
    card?: ITrustCard;
    yandexAccount?: IYandexAccount;
    onEdit?: () => void;
    onDelete?: () => void;
    setActive?: () => void;
    active?: boolean;
    scroogeLink?: string;
}

const DEFAULT_CARD_MASK = '510000****3350';

export class Wallet extends React.Component<IWalletProps, any> {
    render() {
        const { account, onEdit, onDelete, setActive, active, card, yandexAccount, scroogeLink } = this.props;

        let cardBankingDetails = <p className={styles.cardBankingDetails}>{EMPTY_DATA}</p>;

        if (account) {
            if (account.type === AccountType.CARD && account.details?.card_mask) {
                cardBankingDetails = <p className={styles.cardBankingDetails}>
                    {account.details.card_mask === DEFAULT_CARD_MASK
                        ? <span>{account.details.card_mask} <b>(заглушка)</b></span>
                        : account.details.card_mask
                    }
                </p>;
            } else if (account.type === AccountType.WALLET
                && (account.details.expenditure || account.details.next_refresh)) {
                cardBankingDetails = (
                    <p className={styles.cardBankingDetails}>
                        {account.details.expenditure && rubs(account.details.expenditure) || EMPTY_DATA} /&nbsp;
                        <FormatDate value={account.details.next_refresh * ONE_SECOND || EMPTY_DATA}
                                    onlyDate/>
                    </p>
                );
            }
        }

        if (card || yandexAccount) {
            const id = card?.id || yandexAccount?.id;
            cardBankingDetails = <p title={id} className={styles.cardId}>{id}</p>;
        }

        const activeCtrlCn = active ? styles.activeCtrl : styles.blockedCtrl;
        let statusCardCn = '';
        if (card?.card_level) {
            const cardLevel = card.card_level.toLowerCase().split(' ');
            statusCardCn = cardLevel[0] === CardLevels.GOLD ? styles.card_gold
                : cardLevel[0] === CardLevels.WORLD && (cardLevel[1] === CardLevels.WORLD_BLACK
                    || cardLevel[1] === CardLevels.WORLD_ELITE) ? styles.card_black
                    : cardLevel[0] === CardLevels.WORLD ? styles.card_world
                        : cardLevel[0] === CardLevels.INFINITE ? styles.card_black
                            : cardLevel[0] === CardLevels.SIGNATURE ? styles.card_black
                                : cardLevel[0] === CardLevels.PLATINUM ? styles.card_platinum
                                    : '';
        }

        const driveTypeId = 8974;
        const taxiTypeId = 114;
        const b2bIcon = account?.parent?.type_id === driveTypeId
            ? `${styles.b2bIcon} ${styles.b2bIcon_drive}`
            : account?.parent?.type_id === taxiTypeId
                ? `${styles.b2bIcon} ${styles.b2bIcon_taxi}`
                : '';
        const isExpired = card?.expired;
        const isFamilyCard = card?.payer_info || card?.is_family_card_owner;
        const familyInfo = `Семейная карта (${card?.is_family_card_owner ? 'владелец' : 'родственник'})`;
        const familyLimit = card?.payer_info?.family_info?.currency === "RUB"
            ? rubs(card?.payer_info?.family_info?.limit)
            : `${card?.payer_info?.family_info?.limit} ${card?.payer_info?.family_info?.currency}`;
        const familyLimitInfo = isFamilyCard
            ? `${familyLimit} ${FRAME_TRANSLATE[card?.payer_info?.family_info?.frame ?? ''] ?? ''}`
            : null;
        const familyBalance = isFamilyCard
            ? `${rubs((card?.payer_info?.family_info?.limit ?? 0) - (card?.payer_info?.family_info?.expenses ?? 0))}`
            : null;
        const overdraft = Number(account?.parent?.details?.balance_info?.overdraft);

        return (
            <div className={`${styles.card} ${!active && styles.cardBlocked || ''} ${statusCardCn} ${isExpired
                ? `${styles.card_expired}` : ''}`}
                 title={isExpired ? 'Просрочена' : ''}>
                {account ? (
                    <div className={styles.cardHeader}>
                        <p className={styles.cardTitle} title={account.name}>{account.name}</p>
                        <div>
                            <Link className={styles.cardCtrl}
                                  onClick={onEdit ? onEdit.bind(this) : undefined}>
                                edit
                            </Link>
                            <Link className={styles.cardCtrl}
                                  onClick={onDelete ? onDelete.bind(this) : undefined}>
                                del
                            </Link>
                            <Link className={`${styles.cardCtrl} ${activeCtrlCn}`}
                                  onClick={setActive ? setActive.bind(this) : undefined}>
                                {active ? 'active' : 'blocked'}
                            </Link>
                        </div>
                    </div>
                ) : (card?.account || yandexAccount?.account) ? (
                    <div className={styles.cardHeader}>
                        <p className={styles.cardTitle}>
                            <Copy>{card?.account || yandexAccount?.account}</Copy>
                        </p>
                        {scroogeLink
                            ? <Link href={scroogeLink}
                                    target={'_blank'}
                                    className={styles.cardCtrl}>
                                Scrooge
                            </Link>
                            : undefined
                        }
                    </div>
                ) : undefined}
                <p className={styles.cardDetails}>
                    {isFamilyCard
                        ? familyInfo
                        : account?.type || card?.payment_method || yandexAccount?.payment_method || EMPTY_DATA
                    }
                    {account ? ` — ${account.id} ` : undefined}
                    {account?.parent?.id
                        ? <span>
                            — <Link title={`Parent wallet is ${account?.parent?.is_active ? 'active' : 'blocked'}`}
                                    className={styles[`parentId${account?.parent?.is_active ? 'Active' : 'Blocked'}`]}
                                    target={'_blank'}
                                    href={`#/b2b/search-wallets?parent_id=${account.parent.id}`}>
                                p_id: {account.parent.id}
                            </Link>
                        </span>
                        : undefined
                    }

                </p>
                {isFamilyCard && !card?.is_family_card_owner
                    ? <div>{familyBalance ?? EMPTY_DATA} / {familyLimitInfo ?? EMPTY_DATA}</div>
                    : null
                }
                {cardBankingDetails}
                {account
                    ? <>
                        <p className={styles.cardDetails}>
                            {rubs(account.hard_limit)}&nbsp;/&nbsp;{rubs(account.soft_limit)}
                        </p>
                        <div className={`${styles.containerWithIcon}`}>
                            <div className={styles.leftSide}>
                                <p className={styles.cardBalances}>
                                    <span title={'Баланс'}>Б: </span>
                                    <span className={styles.balance}>
                                        {rubs(account?.parent?.balance ?? 0)}
                                    </span>
                                    {overdraft >= 0
                                        && <>&nbsp;/&nbsp;
                                            <span title={'Овердрафт'}>О: </span>
                                            <span className={styles.balance}>
                                                {rubs(overdraft)}
                                            </span>
                                        </>}
                                </p>
                                <p className={styles.cardBalances}>
                                    <span title={'Лимит'}>Л: </span>
                                    <span className={styles.balance}>
                                        {rubs(account.balance)}
                                    </span>
                                    &nbsp;/&nbsp;
                                    <span className={styles.spent}>
                                        {rubs(account.spent)}
                                    </span>
                                </p>
                            </div>
                            <p className={styles.rightSide}>
                                <span className={b2bIcon}>{}</span>
                            </p>
                        </div>
                    </>
                    : card
                        ? <>
                            <div className={`${styles.containerWithIcon} ${styles.issuerDetails}`}>
                                <p className={styles.leftSide}>
                                    {card?.binding_ts && (
                                        <FormatDate className={styles.cardDetails}
                                                    value={card?.binding_ts * ONE_SECOND}/>
                                    )}
                                    <span className={`${styles.issuerDetailsLine}`}
                                          title={card?.card_bank}>
                                        {card?.card_bank}
                                    </span>
                                </p>
                                {card?.system && (
                                    <p className={styles.rightSide}>
                                        <PaymentSystemLogos value={card?.system}/>
                                        <span className={styles.issuerDetailsLine}
                                              title={card?.card_level}>
                                            {card?.card_level}
                                        </span>
                                    </p>
                                )}
                            </div>
                        </>
                        : yandexAccount
                            ? <>
                                <div className={`${styles.containerWithIcon} ${styles.issuerDetails}`}>
                                    <p className={styles.leftSide}>
                                        {yandexAccount?.binding_ts
                                            ? <FormatDate className={styles.cardDetails}
                                                          value={yandexAccount.binding_ts * ONE_SECOND}/>
                                            : undefined}
                                        <p className={`${styles.cardBalances} ${styles.issuerDetailsLine}`}>
                                            Баланс:&nbsp;
                                            <span className={styles.balance}>
                                                {rubs(yandexAccount.balance)}
                                            </span>
                                        </p>
                                    </p>
                                    <p className={styles.rightSide}>
                                        <PaymentSystemLogos value={PaymentSystem.YANDEX_ACCOUNT}/>
                                    </p>
                                </div>
                            </>
                            : undefined
                }
            </div>
        );
    }
}
