import React from 'react';

import { DAYS_LABEL, EMPTY_DATA, ONE_WEEK_IN_DAYS } from '../../constants';
import { Restrictions, TimeRestriction } from '../../models/role';
import { isObjectEqual } from '../../utils/isObjectEqual';
import { Cross } from '../Cross';
import { Window } from '../FullModal';
import { Input } from '../Input';
import { Link } from '../Link';
import * as style from './index.css';

export interface IWeekCalendarPropsItem {
    name: string;
    time_restrictions: TimeRestriction;
}

interface IWeekCalendarProps {
    items: IWeekCalendarPropsItem[];
    onChange?: (items: IWeekCalendarPropsItem[]) => void;
    itemsObj?: any;
}

interface IWeekCalendarState {
    items: IWeekCalendarPropsItem[];
    restrictionItem: IWeekCalendarPropsItem;
    changeTimesIsOpen: boolean;
    restrictionItemIndex: number;
}

export class WeekCalendar extends React.Component<IWeekCalendarProps, IWeekCalendarState> {
    state = {
        items: this.props.items || [],
        restrictionItem: {} as IWeekCalendarPropsItem,
        changeTimesIsOpen: false,
        restrictionItemIndex: 0,
    };

    componentDidMount(): void {
        this.setState({ items: this.props.items || [] });
    }

    componentDidUpdate(
        prevProps: Readonly<IWeekCalendarProps>,
        prevState: Readonly<IWeekCalendarState>,
        snapshot?: any,
    ): void {
        if (!isObjectEqual(prevProps.items, this.props.items)) {
            this.setState({ items: this.props.items || [] });
        }
    }

    addRestriction(name) {
        const items = this.state.items;
        const item = items.find(item => item.name === name);
        let newRestrictions: Restrictions[] = [];

        if (item && !item.time_restrictions) {
            item.time_restrictions = { restrictions: [], time_zone: 3 };
        }

        if (item && item.time_restrictions.restrictions) {
            newRestrictions = [...item.time_restrictions.restrictions, {} as Restrictions];
            item.time_restrictions.restrictions = newRestrictions;
        }

        this.props.onChange && this.props.onChange(items);
    }

    deleteLinkItem(name) {
        const items = this.state.items;
        const _items = items.filter(item => item.name !== name);

        this.props.onChange && this.props.onChange(_items);
    }

    deleteItem(name: string, index: number) {
        const items = this.state.items;
        const item = items.find(item => item.name === name);
        item
        && item.time_restrictions
        && item.time_restrictions.restrictions
        && item.time_restrictions.restrictions.splice(index, 1);
        this.props.onChange && this.props.onChange(items);
    }

    changeDays(name, _item, day) {
        const items = this.state.items;
        const item = items.find(item => item.name === name);
        const settingsDay = item
            && item.time_restrictions
            && item.time_restrictions.restrictions
            && item.time_restrictions.restrictions[_item]
            && item.time_restrictions.restrictions[_item].day || 0;

        let newDays = settingsDay;

        const olreadySelected = (settingsDay & (1 << day)) > 0;

        if (olreadySelected) {
            newDays = (settingsDay & ~(1 << day));
        } else {
            newDays = (settingsDay | (1 << day));
        }

        item
        && item.time_restrictions
        && item.time_restrictions.restrictions
        && item.time_restrictions.restrictions[_item]
        && (item.time_restrictions.restrictions[_item].day = newDays);
        this.props.onChange && this.props.onChange(items);
    }

    changeTime(name, _item, field, value) {
        const items = this.state.items;
        const item = items.find(item => item.name === name);

        if (value === '') {
            item
            && item.time_restrictions
            && item.time_restrictions.restrictions
            && item.time_restrictions.restrictions[_item]
            && (delete item.time_restrictions.restrictions[_item][field]);
        } else {
            item
            && item.time_restrictions
            && item.time_restrictions.restrictions
            && item.time_restrictions.restrictions[_item]
            && (item.time_restrictions.restrictions[_item][field] = (value === '' ? value : +value));
        }

        this.props.onChange && this.props.onChange(items);
    }

    close() {
        this.setState({ changeTimesIsOpen: false });
    }

    openTimeChange(item, index) {
        this.setState({ changeTimesIsOpen: true, restrictionItem: item, restrictionItemIndex: index });
    }

    render() {
        //TODO: check render with state.items
        const row = this.props.items && this.props.items
            .sort((a, b) => a.name.localeCompare(b.name))
            .map((item, index) => {
                const restrictions = item.time_restrictions
                    && item.time_restrictions.restrictions
                    && Array.isArray(item.time_restrictions.restrictions)
                    && item.time_restrictions.restrictions || [];
                const { type, description, link, enabled = true } = this.props.itemsObj?.[item.name] || {};
                const tr = <tr key={index}>
                    <td onClick={this.addRestriction.bind(this, item.name)}>+</td>
                    <td className={style.name}>
                        <div><Link href={link} target={'_blank'}>{item.name}</Link></div>
                        <div className={style.description}>
                            <span className={`${style.type} ${enabled ? style.enabled : style.disabled}`}>
                                {type ? `${type}: ` : ''}
                            </span>{description}
                        </div>
                    </td>
                    <td colSpan={3}><Link onClick={this.deleteLinkItem.bind(this, item.name)}>delete</Link></td>
                </tr>;
                const subtr = restrictions.map((_item, _index) => {
                    return <tr key={`${index}_${_index}`} className={style.day_row}>
                        <td/>
                        <td><Week day={_item.day} onClick={this.changeDays.bind(this, item.name, _index)}/></td>
                        <td>
                            <Link onClick={this.openTimeChange.bind(this, item, _index)}>
                                {_item.time_from || _item.time_from === 0 ? _item.time_from : EMPTY_DATA}
                            </Link>
                            <span className={style.gmt}>GMT+{item.time_restrictions.time_zone}</span>
                        </td>
                        <td>
                            <Link onClick={this.openTimeChange.bind(this, item, _index)}>
                                {_item.time_to || _item.time_to === 0 ? _item.time_to : EMPTY_DATA}
                            </Link>
                            <span className={style.gmt}>GMT+{item.time_restrictions.time_zone}</span>
                        </td>
                        <td><Cross className={style.cross}
                                   onClick={this.deleteItem.bind(this, item.name, _index)}/></td>
                    </tr>;
                }) || [];

                return [tr, ...subtr];
            });

        return <div className={style.component}>
            {this.state.changeTimesIsOpen &&
            <DurationControls item={this.state.restrictionItem}
                              index={this.state.restrictionItemIndex}
                              onClose={this.close.bind(this)}
                              onChange={this.changeTime.bind(this)}/>
            }
            <table>
                <thead></thead>
                <tbody>
                    {
                        row
                    }
                </tbody>
            </table>
        </div>;
    }
}

const DurationControls = React.memo((props: {
    onClose: () => void;
    item: IWeekCalendarPropsItem;
    onChange: () => void;
    index: number;
}) => {
    return <Window onClose={props.onClose} title={`${props.item.name}`}>
        <div>
            <Input value={props.item.time_restrictions.restrictions[props.index].time_from}
                   placeholder={`time_from, HHMM, GMT+${props.item.time_restrictions.time_zone}`}
                   className={style.input}
                   onChange={props.onChange.bind(null, props.item.name, props.index, 'time_from')}/>
            <Input value={props.item.time_restrictions.restrictions[props.index].time_to}
                   placeholder={`time_to, HHMM, GMT+${props.item.time_restrictions.time_zone}`}
                   className={style.input}
                   onChange={props.onChange.bind(null, props.item.name, props.index, 'time_to')}/>
        </div>
    </Window>;
});

const Week = React.memo((props: { day: number; onClick: () => void }) => {
    const week = new Array(ONE_WEEK_IN_DAYS).fill(false);
    const HolidayIndex = 4;

    return <div>{
        week.map((day, index) => {
            const check = (props.day & (1 << index)) > 0;

            return <div key={index}
                        className={`${style.day} ${index > HolidayIndex
                            ? style.holiday
                            : ''} ${check && style.active || ''}`}
                        onClick={props.onClick.bind(null, index)}>
                {DAYS_LABEL[index]}
            </div>;
        })
    }</div>;
});
