import * as React from 'react';

import * as styles from './index.css';

const throttle = (f) => {
    let token: number | null = null;
    let lastArgs: any[] | null = null;

    const invoke = () => {
        f(...(lastArgs ?? []));
        token = null;
    };

    const result = (...args) => {
        lastArgs = args;
        if (!token) {
            token = requestAnimationFrame(invoke);
        }
    };

    result.cancel = () => token && cancelAnimationFrame(token);

    return result;
};

interface IDraggable {
    onMove: (x, y) => void;
    x: number;
    y: number;
}

export class Draggable extends React.PureComponent<IDraggable> {
    relX = 0;
    relY = 0;
    ref;

    constructor(props) {
        super(props);

        this.ref = React.createRef<HTMLDivElement>();
    }

    update = throttle(() => {
        const { x, y } = this.props;

        if (this.ref?.current) {
            this.ref.current.style.transform = `translate(${x}px, ${y}px)`;
        }
    });

    onMouseDown = (event) => {
        const { scrollLeft, scrollTop, clientLeft, clientTop } = document.body;
        const { left = 0, top = 0 } = this.ref?.current?.getBoundingClientRect();

        if (event.button !== 0) {
            return;
        }

        this.relX = event.pageX - (left + scrollLeft - clientLeft);
        this.relY = event.pageY - (top + scrollTop - clientTop);

        if (this.ref?.current) {
            this.ref.current.style.cursor = 'move';
        }

        window.addEventListener('mousemove', this.onMouseMove);
        window.addEventListener('mouseup', this.onMouseUp);

        event.preventDefault();
    };

    onMouseUp = (event) => {
        this.ref.current.style.cursor = 'auto';

        window.removeEventListener('mousemove', this.onMouseMove);
        window.removeEventListener('mouseup', this.onMouseUp);

        event.preventDefault();
    };

    onMouseMove = (event) => {
        this.props.onMove(
            event.pageX - this.relX,
            event.pageY - this.relY,
        );
        event.preventDefault();
    };

    componentDidMount() {
        this.ref?.current?.addEventListener('mousedown', this.onMouseDown);
        this.update();
    }

    componentDidUpdate() {
        this.update();
    }

    componentWillUnmount() {
        this.ref?.current?.removeEventListener('mousedown', this.onMouseDown);
        this.update.cancel();
    }

    render() {
        return (
            <div className={styles.draggable}
                 ref={this.ref}>
                {this.props.children}
            </div>
        );
    }
}
