const width = 500;
const height = 500;

let zoom = width;
const step = 10;

const minZoom = 100;
const mazZoom = 1000;

export default function SVGHelper(svg) {

    svg.setAttribute('width', width.toString());
    svg.setAttribute('height', height.toString());
    svg.setAttribute('viewBox', `0 0 ${width} ${height}`);
    let currentSelected: SVGSVGElement | null = null;
    const connections: Map<any, any> = new Map();

    function mousewheel(e) {
        e.preventDefault();

        if (e.deltaY > 0) {
            zoom += step;
        } else {
            zoom -= step;
        }

        if (zoom < minZoom || zoom > mazZoom) {
            return;
        }

        svg.setAttribute('viewBox', `0 0 ${zoom} ${zoom}`);
    }

    function resetZoom() {
        zoom = width;
        svg.setAttribute('viewBox', `0 0 ${width} ${height}`);
    }

    function mousemove(e) {
        const cursor = getCursorPosition(e, svg);
        if (currentSelected) {
            const { paths } = connections.get(currentSelected.getAttribute('id')) || {};

            currentSelected.setAttribute('cx', cursor.x);
            currentSelected.setAttribute('cy', cursor.y);

            paths?.length && paths.forEach(_path => {
                const { path, label } = _path;
                const stringPath = path.getAttribute('d').split(',');
                let newPath = '';
                if (label === 'from') {
                    newPath = `M${cursor.x} ${cursor.y},${stringPath[1]}`;
                } else {
                    newPath = `${stringPath[0]}, ${cursor.x} ${cursor.y}`;
                }

                path.setAttribute('d', newPath);
            });

        }
    }

    function mousedown(e) {
        if (e.target.nodeName === 'circle') {
            e.target.setAttribute('stroke-width', '2');
            e.target.setAttribute('stroke', 'green');
            currentSelected = e.target;
        }
    }

    function mouseup(e) {
        e.target.setAttribute('stroke-width', '1');
        e.target.setAttribute('stroke', '#b2b2b2');
        currentSelected = null;
    }

    function getCursorPosition(event, svgElement) {
        const svgPoint = svgElement.createSVGPoint();

        svgPoint.x = event.clientX;
        svgPoint.y = event.clientY;

        return svgPoint.matrixTransform(svgElement.getScreenCTM().inverse());
    }

    svg.addEventListener('mousemove', mousemove, false);
    svg.addEventListener('mousedown', mousedown);
    svg.addEventListener('mouseup', mouseup);
    svg.addEventListener('wheel', mousewheel);
    svg.addEventListener('dblclick', resetZoom);

    return {
        log: () => {
            console.info(svg);
        },
        append: (element) => {
            if (Array.isArray(element)) {
                element.forEach(el => svg.appendChild(el));
            } else {
                svg.appendChild(element);
            }
        },
        destroy: () => {
            svg.removeEventListener('mouseup', mouseup);
            svg.removeEventListener('mousedown', mousedown);
            svg.removeEventListener('mousemove', mousemove);
            svg.removeEventListener('wheel', mousewheel);
            svg.removeEventListener('dblclick', resetZoom);
        },
        connect: (connection: [SVGSVGElement, SVGSVGElement][]) => {
            connection.forEach(connection => {
                const X0 = +(connection[0].getAttribute('cx') || 0);
                const X1 = +(connection[1].getAttribute('cx') || 0);
                const Y0 = +(connection[0].getAttribute('cy') || 0);
                const Y1 = +(connection[1].getAttribute('cy') || 0);

                const path = createPath({
                    from: `${X0} ${Y0}`,
                    to: `${X1} ${Y1}`,
                });
                connections.set(connection[0].id, {
                    paths: [...connections.get(connection[0].id)?.paths || [], { path, label: 'from' }],
                });
                connections.set(connection[1].id, {
                    paths: [...connections.get(connection[1].id)?.paths || [], { path, label: 'to' }],
                });

                svg.appendChild(path);
            });
        },
    };
}

export function createCircle({ id, cx, cy, r, fill = 'rgba(255,0,0,0.1)', stroke = '#b2b2b2' }) {
    const circle = document.createElementNS('http://www.w3.org/2000/svg', 'circle');
    circle.setAttribute('id', id);
    circle.setAttribute('cx', cx);
    circle.setAttribute('cy', cy);
    circle.setAttribute('r', r);
    circle.setAttribute('fill', fill);
    circle.setAttribute('stroke', stroke);

    return circle;
}

export function createPath({ from, to }) {
    const path = document.createElementNS('http://www.w3.org/2000/svg', 'path');
    const stringPath = `M${from},${to} `;
    path.setAttribute('d', stringPath);
    path.setAttribute('stroke', '#b2b2b2');
    path.setAttribute('stroke-width', '.2');

    return path;
}
