import React from 'react';

import { Dict } from '../../../types';
import { EMPTY_DATA, LOCALIZATION_RESOURCE_PREFIX } from '../../constants';
import { TabItem, Tabs } from '../../ui/Tabs';
import { INavigationRouteItem } from '../navigation';
import { Translate } from '../translate';

export interface ITabViewProps {
    title?: string;
    withoutTitle?: boolean;
    tabs: INavigationRouteItem[];
    uri: string;
    t: Translate;
    match?: Dict<any>;
    location?: any;
}

interface ITabViewState {
    currentTab: string;
    tabs: TabItem[];
}

export class TabView extends React.Component<ITabViewProps, ITabViewState> {
    state: ITabViewState = {
        currentTab: '',
        tabs: [] as TabItem[],
    };

    constructor(props: ITabViewProps) {
        super(props);
        this.state = this.getData();
    }

    componentDidUpdate(prevProps: Readonly<ITabViewProps>): void {
        if (prevProps.title !== this.props.title
            || prevProps.withoutTitle !== this.props.withoutTitle
            || prevProps.tabs !== this.props.tabs
            || prevProps.uri !== this.props.uri) {
            this.setState(this.getData());
        }
    }

    getData(): { tabs: TabItem[]; currentTab: string } {
        const { cgi, linkItems, currentPageTabIndex } = this.getLinkStructure();

        const currentPageTab = linkItems[currentPageTabIndex];

        const tabs: TabItem[] = this.props.tabs && Array.isArray(this.props.tabs)
            ? this.props.tabs
                .map(tab => {
                    return { name: tab.uri || EMPTY_DATA, link: tab.uri || '/', order: tab.order };
                })
            : [];
        const currentTab = tabs.map(tab => tab.link).includes(currentPageTab)
            ? currentPageTab
            : tabs[0].link;

        const newLinkItems = [...linkItems];
        newLinkItems[currentPageTabIndex] = currentTab;

        location.hash = this.buildHash(newLinkItems, cgi);

        return { tabs, currentTab };
    }

    selectTab(currentTab: string): void {
        this.setState({ currentTab }, () => {
            const { cgi, linkItems, currentPageTabIndex } = this.getLinkStructure();

            const newLinkItems = [...linkItems];
            newLinkItems[currentPageTabIndex] = currentTab;
            newLinkItems.splice(currentPageTabIndex + 1);

            location.hash = this.buildHash(newLinkItems, cgi);
        });
    }

    getLinkStructure() {
        const fullHash = location.hash && location.hash.split('?');
        const cgi = fullHash[1];
        const linkItems = fullHash[0].replace(/^#?\//ig, '')
            .replace(/\/$/ig, '').split('/');

        const linkUriItems = this.props.uri && this.props.uri.split('?')[0].replace(/^#?\//ig, '')
            .replace(/\/$/ig, '').split('/');

        const currentPageTabIndex = linkUriItems.length;

        return { cgi, linkItems, currentPageTabIndex };
    }

    buildHash(linkItems: string[], cgi?: string) {
        return `#/${linkItems.join('/')}${cgi ? `?${cgi}` : ''}`;
    }

    render() {
        const { t, title } = this.props;

        return <div>
            {!this.props.withoutTitle
                ? <h1>{t.getItem(LOCALIZATION_RESOURCE_PREFIX + title)}</h1>
                : null}
            <Tabs tabs={this.state.tabs}
                  t={t}
                  currentTab={this.state.currentTab}
                  selectTab={this.selectTab.bind(this)}/>
        </div>;
    }
}
