import Dexie from 'dexie';

import { LSSettingItems } from '../../types';
import LS from './localStorage/localStorage';

interface ICacheProps {
    id: string;
    ttl?: number;
    reduceResponse?: (data: ICacheData) => any;
    onCache: (data: ICacheData) => {};
    onResponse: (data: ICacheData) => {};
    onError: (data: ICacheData) => {};
}

interface ICacheData {

}

interface ICacheExecProps {
    request: Function;
    requestKey: string;
}

const DEFAULT_TTL = 10000;
const DEFAULT_REDUCE = (data) => data;

class CacheHelper {
    private id: string;
    private ttl: number;
    private reduceResponse: (data: ICacheData) => any
    private onCache: (data: ICacheData) => {};
    private onResponse: (data: ICacheData) => {};
    private onError: (data: ICacheData) => {};
    private db: RequestCache;
    private ls;

    constructor(props: ICacheProps) {
        this.id = props.id;
        this.ttl = props.ttl ?? DEFAULT_TTL;
        this.onCache = props.onCache.bind(this);
        this.onResponse = props.onResponse.bind(this);
        this.onError = props.onError.bind(this);
        this.db = new RequestCache();
        this.ls = new LS();
        this.reduceResponse = props?.reduceResponse ?? DEFAULT_REDUCE;
    }

    async exec(props: ICacheExecProps) {
        const data = await this.db.requestCache
            .where('[request+env]')
            .equals([props.requestKey, this.ls.get(LSSettingItems.env)])
            .toArray();

        this.onCache(data[0]);

        props.request()
            .then(response => {
                const result = this.reduceResponse(response);

                this.onResponse(result);

                this.db.requestCache.put({
                    request: this.id,
                    timestamp: +new Date(),
                    env: this.ls.get(LSSettingItems.env),
                    ttl: this.ttl,
                    response: result,
                });
            })
            .catch((error) => {
                this.onError(error);
            });
    }
}

interface IRequestCacheTable {
    request: string;
    timestamp: number;
    env: string;
    ttl: number;
    response: unknown;
}

class RequestCache extends Dexie {
    requestCache: Dexie.Table<IRequestCacheTable, number>;

    constructor() {
        super('requestCache');
        this.version(1).stores({
            requestCache: '&[request+env]',
        });
    }
}

export { CacheHelper };
