import React, { ReactElement, useState } from 'react';
import ReactTooltip from 'react-tooltip';

import EngineIcon from '../../../svg-components/engine.component.svg';
import FuelIcon from '../../../svg-components/fuel.component.svg';
import MileageIcon from '../../../svg-components/mileage.component.svg';
import SpeedIcon from '../../../svg-components/speed.component.svg';
import VoltageIcon from '../../../svg-components/voltage.component.svg';
import { EMPTY_DATA } from '../../constants';
import CarInfo, { CarInfoHandler } from '../../models/car';
import { deepCopy } from '../utils';
import { CarFuelModal } from './CarFuelModal';
import { NORMAL_VOLTAGE, SECOND_LEVEL_VOLTAGE, WARNING_VOLTAGE } from './constants';
import style from './index.css';

const generateIcon = (type, params) => {
    switch (type) {
    case 'engine':
        return <EngineIcon {...params}/>;
    case 'fuel':
        return <FuelIcon {...params}/>;
    case 'mileage':
        return <MileageIcon {...params}/>;
    case 'speed':
        return <SpeedIcon {...params}/>;
    case 'voltage':
        return <VoltageIcon {...params}/>;
    default:
        return null;
    }
};

export interface ICarIndicatorsBlockProps {
    carInfo: CarInfo;
    wrapperStyle: string;
    itemStyle: string;
    iconStyle: string;

    hasTooltip?: boolean;
    allItems?: boolean;
}

export const CarIndicatorsBlock = React.memo((props: ICarIndicatorsBlockProps) => {
    const { carInfo, wrapperStyle, itemStyle, iconStyle, hasTooltip, allItems } = props;
    const indicators: ICarIndicator[] = getCarIndicators(carInfo, allItems);
    const [isFuelModalOpen, openFuelModal] = useState<boolean>(false);
    const carInfoCopy = deepCopy(carInfo);

    return <div className={wrapperStyle}>
        {indicators.map(indicator => {
            return <div key={indicator.id}
                        className={itemStyle}
                        data-tip={hasTooltip}
                        data-for={indicator.id}
                        onClick={indicator.id === 'fuel' ? openFuelModal.bind(null, true) : null}>

                {generateIcon(indicator.id, { className: iconStyle })}

                {indicator.value}

                {hasTooltip
                    ? <ReactTooltip id={indicator.id} place="bottom">
                        <div>{indicator.description}</div>
                    </ReactTooltip>
                    : null
                }
            </div>;
        })}

        {isFuelModalOpen
            ? <CarFuelModal carId={carInfoCopy?.id}
                            imei={carInfoCopy?.imei}
                            onClose={openFuelModal.bind(null)}/>
            : null
        }
    </div>;
});

export interface ICarIndicator {
    id: string;
    description: string;
    value: ReactElement;
}

export function getCarIndicators(carInfo: CarInfo, allIndicators?: boolean): ICarIndicator[] {
    const {
        accLevel,
        voltageCurrentLevel,
        extLevel,
        voltageLevel,
        fuelType,
        secondFuelType,
        fuelLevel,
        secondFuelLevel,
        isEngineOn,
        mileage,
        speed,
    } = getTelematicValues(carInfo);

    const indicators = [
        {
            id: 'voltage',
            description: 'Заряд аккумулятора',
            value: <span>
                <span className={`${style[`voltage_current_level_${voltageCurrentLevel}`]}`}>
                    {accLevel ?? '?'}
                </span>
                <span className={allIndicators ? style.separator : style.tooltip_text}> / </span>
                <span className={`${style[`voltage_level_${voltageLevel}`]} ${style.voltage_second}`}>
                    {extLevel ?? '?'}
                </span>
            </span>,
        },
        {
            id: 'fuel',
            description: 'Топливо',
            value: <span>
                <span>{fuelLevel ?? '? '}%</span>
                {fuelType
                    ? <span className={allIndicators ? style.low_level_text : style.tooltip_text}> ({fuelType})</span>
                    : null
                }
                {secondFuelType === 'gas'
                    ? <>
                        <span className={style.low_level_text}> | </span>
                        <span>{secondFuelLevel ?? '? '}%</span>
                        <span className={allIndicators ? style.low_level_text : style.tooltip_text}>
                            ({secondFuelType})
                        </span>
                    </>
                    : null
                }
            </span>,
        },
        {
            id: 'engine',
            description: 'Двигатель',
            value: <span>{isEngineOn ? 'Вкл.' : 'Выкл.'}</span>,

        },
        {
            id: 'speed',
            description: 'Скорость',
            value: <span>{speed ? `${speed} км/ч` : EMPTY_DATA}</span>,
        },
    ];

    if (allIndicators) {
        const placeToPast = 3;
        indicators.splice(placeToPast, 0, {
            id: 'mileage',
            description: 'Пробег',
            value: <span>{mileage ? `${mileage} км` : EMPTY_DATA}</span>,
        });
    }

    return indicators;
}

export interface IGetTelematicValues {
    accLevel: number | null;
    voltageCurrentLevel: VOLTAGE_CURRENT_TYPE;
    extLevel: number | null;
    voltageLevel: EXT_VOLTAGE_VALUES;
    fuelType: string | null;
    secondFuelType: string | null;
    fuelLevel: number | null;
    secondFuelLevel: number | null;
    isEngineOn: boolean;
    mileage: number | null;
    speed: number | null;
}

type EXT_VOLTAGE_VALUES = 'normal' | 'warning' | 'critical';
type VOLTAGE_CURRENT_TYPE = 'normal' | 'critical';

export function getTelematicValues(carInfo: CarInfo): IGetTelematicValues {
    const accLevel = CarInfoHandler.getAccLevel.call(carInfo);
    const voltageCurrentLevel = extVoltageCurrentLevel(+accLevel);
    const extLevel = CarInfoHandler.getExtLevel.call(carInfo);
    const voltageLevel = extVoltageLevel(+extLevel);
    const fuelType = CarInfoHandler.getFuelType.call(carInfo);
    const secondFuelType = CarInfoHandler.getSecondFuelType.call(carInfo);
    const fuelLevel = CarInfoHandler.getFuelLevel.call(carInfo);
    const isEngineOn = CarInfoHandler.isEngineOn.call(carInfo);
    const mileage = CarInfoHandler.getMileage.call(carInfo);
    const speed = CarInfoHandler.getSpeed.call(carInfo);

    let secondFuelLevel = null;

    if (secondFuelType === 'gas') {
        secondFuelLevel = CarInfoHandler.getSecondFuelLevel.call(carInfo);
    }

    return {
        accLevel,
        voltageCurrentLevel,
        extLevel,
        voltageLevel,
        fuelType,
        secondFuelType,
        fuelLevel,
        secondFuelLevel,
        isEngineOn,
        mileage,
        speed,
    };
}

export const extVoltageLevel = (v: number): EXT_VOLTAGE_VALUES => {
    return v >= NORMAL_VOLTAGE || !v
        ? 'normal'
        : (v >= WARNING_VOLTAGE && v < NORMAL_VOLTAGE)
            ? 'warning'
            : 'critical';
};

export const extVoltageCurrentLevel = (v: number): VOLTAGE_CURRENT_TYPE => {
    return v >= SECOND_LEVEL_VOLTAGE || !v
        ? 'normal'
        : 'critical';
};
