import { ENVIRONMENT, LSDefaultSettings, LSSettingItems, ValueOf } from '../../../types';
import { LS_DEFAULT_SETTINGS } from './types';

interface ILS {
    version: number;
}

export class LS {
    name = '_da_ls';
    version = 1;
    _name = `${this.name}_${this.version}`;
    default = LS_DEFAULT_SETTINGS;

    constructor(props?: ILS) {
        if (props) {
            this.version = props.version;
        }
    }

    static compareFields(prevStore: string | null, newStore: string | null, field: LSSettingItems): boolean {
        return LS.getField(prevStore, field) === LS.getField(newStore, field);
    }

    static getField(store: string | null, field: LSSettingItems) {
        let obj: LSSettingItems = {} as LSSettingItems;
        try {
            obj = store && JSON.parse(store);
        } catch (e) {

        }

        return obj?.[field];
    }

    getName(): string {
        return this._name;
    }

    _checkIfExist(): string | null {
        const data = window.localStorage && window.localStorage.getItem(this._name) || null;

        if (!data) {
            this.reset();
        }

        return data;
    }

    init() {
        this._checkIfExist();
    }

    getAllData() {
        let data = this._checkIfExist();

        try {
            data = typeof data === 'string' && JSON.parse(data) || {};
        } catch (e) {
            throw Error('LS: not valid data');
        } finally {
        }

        return data;
    }

    set(field: LSSettingItems, value: ValueOf<LSDefaultSettings>) {
        const data: any = this.getAllData();
        data[field] = value;
        try {
            window.localStorage && window.localStorage.setItem(this._name, JSON.stringify(data));
        } catch (e) {
            throw Error('LS: data save problem');
        } finally {
        }

        return data;
    }

    get(field: LSSettingItems) {
        let data: any = this.getAllData();
        if (!data) {
            throw Error(`LS: data is empty`);
        }

        if (data.hasOwnProperty(field)) {
            data = data[field];
        } else {
            if (LS_DEFAULT_SETTINGS.hasOwnProperty(field)) {
                data = LS_DEFAULT_SETTINGS[field];
            } else {
                throw Error(`LS: not found field <${field}> in settings store.\n\r`);
            }
        }

        return data;
    }

    envIsTesting(): boolean {
        return [ENVIRONMENT.TESTING, ENVIRONMENT.QA, ENVIRONMENT.ST].includes(this.get(LSSettingItems.env));
    }

    envIsBasicTesting(): boolean {
        return [ENVIRONMENT.TESTING, ENVIRONMENT.QA].includes(this.get(LSSettingItems.env));
    }

    envIsProdPre(): boolean {
        return [ENVIRONMENT.PRESTABLE, ENVIRONMENT.ADMIN].includes(this.get(LSSettingItems.env));
    }

    envIsScooter(): boolean {
        return [ENVIRONMENT.SCOOTER, ENVIRONMENT.SCOOTER_TEST].includes(this.get(LSSettingItems.env));
    }

    envIsDM(): boolean {
        return [ENVIRONMENT.DM_TEST,
            ENVIRONMENT.DM_PRESTABLE,
            ENVIRONMENT['LEASING-CABINET']].includes(this.get(LSSettingItems.env));
    }

    reset() {
        window.localStorage && window.localStorage.setItem(this._name, JSON.stringify(this.default));
    }
}

export class MessagesByTopic extends LS {
    _messages = {};
    _key = `${this.name}_mbt`;

    _getAll() {
        let data = localStorage.getItem(this._key);
        if (!data) {
            data = '{}';
            localStorage.setItem(this._key, data);
        }

        try {
            this._messages = JSON.parse(data);
        } catch (e) {
            this._messages = {};
        }

        return this._messages;
    }

    constructor() {
        super();
        this._getAll();
    }

    remove(id) {
        const data: any = this._getAll();
        delete data[id];
        localStorage.setItem(this._key, JSON.stringify(data));
    }

    setText(props: { text: string; id: string }) {
        const data: any = this._getAll();
        if (!data.hasOwnProperty(props.id)) {
            data[props.id] = '';
        }

        data[props.id] = props.text;
        localStorage.setItem(this._key, JSON.stringify(data));
    }

    getText(id: string) {
        const data: any = this._getAll();

        return data[id] || null;
    }
}

export default LS;
