import { Dict, ENVIRONMENT, EXTRA_ENV, LSSettingItems } from '../../../types';
import { ErrorSource, logError } from '../../components/Content/initErrorCounter';
import { ErrorLog } from '../../components/ErrorLogger/db';
import LS from '../localStorage/localStorage';

export enum REQUEST_METHOD {
    GET = 'get',
    POST = 'post',
    PATCH = 'PATCH',
    DELETE = 'DELETE'
}

export const EMPTY_DATA_STATUS_CODE = 204;

export enum API_PREFIXES {
    FRONT = 'front/',
    YQL = '/yql/',
    SAMSARA = '/samsara/',
    SAAS_STAFF = '/saasnew/api/staff/',
    SUPPORT_STAFF = '/saasnew/api/support_staff/',
    HITMAN = '/hitman-execution/',
    YT = '/yt/',
    ANALYTICS = '/analytics/',
    ANALYTICS_NEW = '/saasnew/analytics/',
    ANALYTICS_PRESTABLE = '/dajr-prestable/',
    ANALYTICS_TESTING = '/dajr-testing/',
    _OLD_PY_V1 = '/admin/v1/',
    _OLD_PY_REQUEST_AGGREGATOR = '/request_aggregator/v1/',
    _OLD_PY_KNOW_BASE = '/knowledge_base/v1/',
    B2B = '/saasnew/b2b/',
    CHATS_API_TEST = '/chat-api-test/',
    ST_API = '/st-api/',
    IDM_API = '/idm-api/',
}

export const AbortError = 'AbortError';

const buildPrefix = (pref: API_PREFIXES, env: ENVIRONMENT) => {
    if (pref === API_PREFIXES.ANALYTICS) {
        if (env === ENVIRONMENT.ADMIN) {
            return API_PREFIXES.ANALYTICS;
        }

        if (env === ENVIRONMENT.PRESTABLE) {
            return API_PREFIXES.ANALYTICS_PRESTABLE;
        }

        return API_PREFIXES.ANALYTICS_TESTING;
    }

    if (pref === API_PREFIXES._OLD_PY_V1
        || pref === API_PREFIXES._OLD_PY_REQUEST_AGGREGATOR
        || pref === API_PREFIXES._OLD_PY_KNOW_BASE) {

        if (env === ENVIRONMENT.TESTING) {
            return `/p-testing${pref}`;
        }

        if (env === ENVIRONMENT.QA) {
            return `/p-qa${pref}`;
        }

        return `/py${pref}`;
    }

    return pref;
};

interface IRequestConfig {
    api: string;

    method?: REQUEST_METHOD;
    cgi?: string;
    cgiObj?: {
        [key: string]: any;
    };
    apiPrefix?: API_PREFIXES;
    timeout?: number;
    fake?: boolean;
    fakeData?: any;
}

export interface IComponentRequest {
    [key: string]: IRequestConfig;
}

const rundom = 100000000000;
export const request_id = (username?: string) => {
    return `reqid=${new Date().getTime()}-${(Math.random() * rundom)
        .toFixed(0)}-${/*getCookie('yandex_login')*/username || 'user_name'}-ADMIN`;
};

enum REQUEST_PREPARE {
    JSON = 'json',
    BLOB = 'blob',
    TEXT = 'text'
}

const ADM_BUILD_HEADER = '1';

export interface IRequest2Options {
    environment?: ENVIRONMENT;
    method?: REQUEST_METHOD;
    prepare?: REQUEST_PREPARE;
    username?: string;
    url?: string;
}

export enum specialQueryParamsKeys {
    FORCE = 'force',
    FORCE_ABSENT_KEY = 'FORCE_ABSENT_KEY'
}

interface IRequestOptions {
    queryParams?: any;
    specialQueryParams?: any;
    body?: any;
    headers?: any;
    file?: any;
    withCode?: boolean;
    removeExtraCgi?: boolean;
    blob?: boolean;
    keepalive?: boolean;
    formData?: FormData;
}

export class Request2 {
    environment: ENVIRONMENT = ENVIRONMENT.TESTING;
    method: REQUEST_METHOD = REQUEST_METHOD.GET;
    prepare: REQUEST_PREPARE = REQUEST_PREPARE.JSON;
    requestConfigs: IComponentRequest = {} as IComponentRequest;
    username = '';
    url = '';

    errorLog = new ErrorLog();
    controller = {} as AbortController;

    getExtra() {
        return EXTRA_ENV[this.environment.toUpperCase()] || null;
    }

    headers(options?: IRequestOptions) {
        const extraEnv = this.getExtra();
        let url: string;
        if (this.url) {
            url = this.url;
        } else if (extraEnv?.custom_env) {
            url = `${extraEnv.custom_env}`;
        } else {
            url = `https://${extraEnv?.env || this.environment}.carsharing.yandex.net`;
        }

        const contentType = options?.formData ? {} : { 'Content-Type': 'application/json' };

        return {
            saasnew: url,
            admbuild: process.env.VERSION?.replace(/[.-]/ig, '') || ADM_BUILD_HEADER,
            ...contentType,
        };
    }

    updateController() {
        try {
            this.controller = window?.AbortController ? new window.AbortController() : {} as AbortController;
        } catch (e) {
            this.controller = {} as AbortController;
        }
    }

    constructor(props: { requestConfigs: IComponentRequest }, options?: IRequest2Options) {
        this.updateController();
        this.requestConfigs = props.requestConfigs;
        this.setOptions(options);
    }

    setOptions(options?: IRequest2Options) {
        try { // problem with initial on webworker TODO: try globalThis for checking window
            const ls = new LS();
            this.environment = ls.get(LSSettingItems.env);
            this.url = ls.get(LSSettingItems.env) === ENVIRONMENT.CUSTOM ? ls.get(LSSettingItems.custom_env) : '';
        } catch (err) {
        }

        if (options) {
            options.environment && (this.environment = options.environment);
            options.method && (this.method = options.method);
            options.prepare && (this.prepare = options.prepare);
            options.username && (this.username = options.username);
            if (options.url && options.environment) {
                this.url = (options.environment === ENVIRONMENT.CUSTOM) ? options.url : '';
            }
        }

        return this;
    }

    error(error) {}

    matchQueryParams(str: string, options: any): string {
        options && options.queryParams && Object.entries(options.queryParams).map((param: any[]) => {
            const reqExp = new RegExp('\\${' + param[0] + '}', 'ig');
            str = str.replace(reqExp, param[1]);
        });

        return str;
    }

    matchCgiParams(cgiObj: any, options): string {
        const queryParams: any = options && options.queryParams;
        const specialQueryParams: Dict<{key: specialQueryParamsKeys; value: unknown}> = options
            && options.specialQueryParams;

        const cgi: string[] = Object.entries(cgiObj).reduce((_p: string[], _c: [string, {value: unknown}]) => {
            if (_c[1]) {
                _p.push(`${_c[0]}=${_c[1]}`);
            } else if (_c[1] === null) {
                if (queryParams && queryParams.hasOwnProperty(_c[0]) && queryParams[_c[0]]) {
                    _p.push(`${_c[0]}=${queryParams[_c[0]]}`);
                }

                if (specialQueryParams && specialQueryParams.hasOwnProperty(_c[0]) && specialQueryParams[_c[0]]) {
                    const queryObj = specialQueryParams[_c[0]];
                    if (queryObj.key === specialQueryParamsKeys.FORCE) {
                        _p.push(`${_c[0]}=${queryObj.value}`);
                    }
                }
            }

            return _p;
        }, []);

        const extraForcedCgi: string[] = specialQueryParams
            ? Object.entries(specialQueryParams)
                ?.reduce((acc: string[], entry ) => {
                    if (entry[1]?.key === specialQueryParamsKeys.FORCE_ABSENT_KEY) {
                        acc.push(`${entry[0]}=${entry[1]?.value}`);
                    }

                    return acc;
                }, [])
            : [];

        return cgi.concat(extraForcedCgi).join('&');
    }

    abort() {
        this.controller?.abort?.();
    }

    exec<T = any>(request: string, options?: IRequestOptions): Promise<T> {
        this.updateController();
        if (this.requestConfigs && this.requestConfigs.hasOwnProperty(request)) {
            const props = this.requestConfigs[request];
            const {
                body = null,
                file = null,
                withCode = false,
                removeExtraCgi,
                keepalive = false,
                formData,
            } = options || {};
            const pref = props.apiPrefix || API_PREFIXES.SAAS_STAFF;
            const apiPrefix = buildPrefix(pref, this.environment);
            const endpoint = `${apiPrefix}${this.matchQueryParams(props.api, options)}`;
            const cgi = `${props.cgiObj && ('&' + this.matchCgiParams(props.cgiObj, options)) || ''}`;

            let extraEnvCgi = '';
            if (!removeExtraCgi) {
                const extraEnv = this.getExtra();
                if (extraEnv) {
                    extraEnvCgi = `&${(new URLSearchParams(extraEnv.cgi)).toString()}`;
                }
            }

            const url = endpoint
                + `?${request_id()}`
                + cgi
                + extraEnvCgi;
            const method = props.method || this.method;

            return fetch(url
                , {
                    signal: this.controller?.signal,
                    method,
                    headers: Object.assign({}, this.headers(options), options && options.headers || {}),
                    referrerPolicy: 'origin-when-cross-origin',
                    credentials: 'include',
                    keepalive,
                    ...body && { body: JSON.stringify(body) },
                    ...file && { body: file },
                    ...formData && { body: formData },
                })
                .then(async response => {
                    if (!response.ok) {
                        let __response = Promise.resolve({});
                        try {
                            __response = Promise.resolve(await response.json());
                        } catch (error) {
                            __response = Promise.resolve({
                                response: {
                                    status: response.status,
                                    error_details: {
                                        special_info: response.statusText,
                                    },
                                },
                            });
                        }

                        return __response.then((data) => {
                            const config = {
                                url, headers: this.headers(),
                                endpoint,
                                method,
                            };
                            const _response = {
                                data, config,
                                status: response.status,
                            };
                            const request = {
                                body,
                                options: options && options.queryParams || null,
                            };
                            const error = {
                                data,
                                config,
                                response: _response,
                                request,
                                env: this.environment,
                            };
                            logError(
                                new Error(JSON.stringify({ endpoint, status: response.status, env: this.environment })),
                                ErrorSource.BACKEND_RESPONSE,
                                { fullError: error },
                            );
                            this.errorLog.logError(error);
                            throw error;
                        });
                    }

                    return response;
                })
                .then(async response => {
                    if (options?.blob) {
                        return response.blob();
                    }

                    const textString = await response.text();
                    try {
                        const data = JSON.parse(textString);
                        if (withCode) {
                            return [data, response.status];
                        }

                        return data;
                    } catch (err) {
                        if (withCode) {
                            return [textString, response.status];
                        }

                        return textString;
                    }
                })
                .catch(error => {
                    throw error;
                });

        }

        return Promise.reject(new Error(`Don't have request handler ${request}`));

    }

    //TODO remove
    execRaw<T = any>(url: string, options?: IRequestOptions): Promise<T> {

        this.updateController();
        // if (this.requestConfigs && this.requestConfigs.hasOwnProperty(request)) {
        //     let props = this.requestConfigs[request];
        //     let {body = null, file = null, withCode = false, removeExtraCgi} = options || {};
        //     let pref = props.apiPrefix || API_PREFIXES.SAAS_STAFF;
        //     let apiPrefix = buildPrefix(pref, this.environment);
        //     let endpoint = `${apiPrefix}${this.matchQueryParams(props.api, options)}`;
        //     let cgi = `${props.cgiObj && ('&' + this.matchCgiParams(props.cgiObj, options)) || ''}`;
        //
        //     let extraEnvCgi = '';
        //     if (!removeExtraCgi) {
        //         let extraEnv = this.getExtra();
        //         if (extraEnv) {
        //             extraEnvCgi = `&${(new URLSearchParams(extraEnv.cgi)).toString()}`;
        //         }
        //     }

        // let url = endpoint
        //     + `?${request_id()}`
        //     + cgi
        //     + extraEnvCgi;
        // let method = props.method || this.method;
        return fetch(url
            , {
                signal: this.controller?.signal,
                method: 'get',
                headers: Object.assign({}, this.headers(), options && options.headers || {}),
                referrerPolicy: 'origin-when-cross-origin',
                credentials: 'include',
                // ...body && {body: JSON.stringify(body)},
                // ...file && {body: file}
            })
            .then(async response => {
                if (!response.ok) {
                    // let __response = Promise.resolve({});
                    // try {
                    //     __response = Promise.resolve(await response.json());
                    // } catch (error) {
                    //     __response = Promise.resolve({
                    //         response: {
                    //             status: response.status,
                    //             error_details: {
                    //                 special_info: response.statusText
                    //             }
                    //         }
                    //     });
                    // }

                    // return __response.then((data) => {
                    //     let config = {
                    //         url, headers: this.headers(),
                    //         endpoint,
                    //         method
                    //     };
                    //     let _response = {
                    //         data, config,
                    //         status: response.status
                    //     };
                    //     let request = {
                    //         body,
                    //         options: options && options.queryParams || null
                    //     };
                    //     let error = {
                    //         data,
                    //         config,
                    //         response: _response,
                    //         request,
                    //         env: this.environment
                    //     };
                    //     this.errorLog.logError(error);
                    //     throw error;
                    // });
                }

                return response;
            })
            .then(async response => {
                const textString = await response.text();
                try {
                    const data = JSON.parse(textString);

                    // if (withCode) {
                    //     return [data, response.status];
                    // }
                    return data;
                } catch (err) {
                    // if (withCode) {
                    //     return [textString, response.status];
                    // }
                    return textString;
                }
            })
            .catch(error => {
                throw error;
            });

        // } else {
        //     return Promise.reject(new Error(`Don't have request handler ${request}`));
        // }
    }
}

export const combineRequests = (...requests) => {
    const combineRequest = {};
    requests.forEach(_r => {
        Object.entries(_r).forEach(__r => {
            Object.assign(combineRequest, { [__r[0]]: __r[1] });
        });
    });

    return combineRequest;
};
