import { getCLS, getFCP, getFID, getLCP, getTTFB, Metric, NavigationTimingPolyfillEntry } from 'web-vitals';

import { Dict } from '../../../types';
import { getTabName } from '../getTabName';
import { ytLogs } from '../sendLogs';
import { EVENT_TYPES } from '../sendLogs/eventTypes/eventTypes';

// No interface was available by default
interface LargestContentfulPaint extends PerformanceEntry {
    renderTime: DOMHighResTimeStamp;
    loadTime: DOMHighResTimeStamp;
    size: number;
    id: string;
    url: string;
    element?: Element;
}

const getEstimatedLoadTime = (metric: Metric): number | undefined => {
    const entries: any[] = metric?.entries;

    if (entries[0]) {
        /*
            if `entries` is PerformanceNavigatorTiming object
            https://www.w3.org/TR/navigation-timing-2/#sec-PerformanceNavigationTiming
        */
        if (entries[0]?.startTime >= 0) {
            const l2Entry: NavigationTimingPolyfillEntry = entries[0];

            return l2Entry.loadEventStart - l2Entry.startTime;
        }

        /*
                if `entries` is navigator.performance.timing object
            */
        if (entries[0]?.navigationStart >= 0) {
            const deprecatedEntry: PerformanceTiming = entries[0];

            return deprecatedEntry.loadEventStart - deprecatedEntry.navigationStart;
        }

    }

    return;
};

const getLCPElement = (metric: Metric): string | undefined => {
    const entries: any[] = metric?.entries;
    const lastEntry: LargestContentfulPaint = entries?.[entries?.length - 1];

    if (lastEntry?.element) {
        return lastEntry.element.outerHTML;
    }

    return;
};

const getLCPElementClassName = (metric: Metric): string | undefined => {
    const entries: any[] = metric?.entries;
    const lastEntry: LargestContentfulPaint = entries?.[entries?.length - 1];

    if (lastEntry?.element) {
        return lastEntry.element.className;
    }

    return;
};

const getDataForPerfLog = (metric) => {
    const initialData = {
        keepalive: true,
        data: {
            event_type: EVENT_TYPES.PERFORMANCE_SNAPSHOT,
            metric: metric.name,
            value: metric.value,
            page: location.href,
            version: process.env.VERSION,
            tabName: getTabName(location.hash.split("#")?.[1].split('?')?.[0]),
        },
    };

    const additionalData: Dict<unknown> = {};

    switch (metric.name) {
    case 'TTFB':
        additionalData.estimatedLoadTime = getEstimatedLoadTime(metric);
        break;
    case 'LCP':
        additionalData.lcpElement = getLCPElement(metric);
        additionalData.lcpElementClassName = getLCPElementClassName(metric);
        break;
    }

    initialData.data = { ...initialData.data, ...additionalData };

    return initialData;
};

export const sendPerfMetrics = () => {
    const log = ytLogs.getInstance();

    const sendToYT = (metric: Metric) => {
        log.send(getDataForPerfLog(metric));
    };

    getTTFB(sendToYT);
    getCLS(sendToYT);
    getFCP(sendToYT);
    getLCP(sendToYT);
    getFID(sendToYT);
};
