/* eslint-disable prefer-rest-params */
import * as React from 'react';

import { Dict, RUNNING_MODE } from '../../types';
import { EMPTY_DATA, ONE_SECOND } from '../constants';
import { UserInfoHandler } from '../models/user';
import FormatDate from '../ui/FormatDate';
import { Link } from '../ui/Link';

declare let ymaps: any;

export const isOffline = (): boolean => {
    return process.env.RUNNING_MODE === RUNNING_MODE.OFFLINE;
};

export const generateuuid4 = () => {
    const RADIX = 16;
    const COOF_THREE = 0x3;
    const COOF_EIGHT = 0x8;

    return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, (c) => {
        const r = Math.random() * RADIX | 0;
        const v = c == 'x' ? r : (r & COOF_THREE | COOF_EIGHT);

        return v.toString(RADIX);
    });
};

export const generateRange = (min: number, max: number, step: number) => {
    const result: number[] = [];
    let value = min;
    while (value <= max) {
        result.push(value);
        value += step;
    }

    return result;
};

export const download = (text: any, name: any, type: any, cb?: any) => {
    const a: HTMLAnchorElement = document.createElement('a');
    document.body.appendChild(a);
    const file: any = new Blob(['\ufeff', text], { type: type });
    a.href = URL.createObjectURL(file);
    a.download = name;
    a.click();
    a.remove();
    cb && cb();
};

export const parseUA = (ua: string) => {
    const r_device = new RegExp('(\\(.*)', 'ig');
    const r_version = new RegExp('.*\\/{1}(\\d|\\.)+', 'ig');

    let version: any = ua.match(r_version);
    let device: any = ua.match(r_device);

    version = version && version[0].split('/')[1];
    device = device && device[0].replace(/(\(|\))/ig, '');

    if (ua.indexOf('CFNetwork') > -1) {
        device = ua;
        version = '';
    }

    const obj = {
        version,
        device,
        raw: ua,
    };

    return obj;
};

const checkBool = (value: any) => {
    if (value == 'true') {
        value = true;
    }

    if (value == 'false') {
        value = false;
    }

    return value;
};

export const parseLocationQuery = (locationQuery: string) => {
    const search: any = {};
    locationQuery.substring(locationQuery.indexOf('?') + 1).split('&')
        .map(item => item.split('='))
        .map(item => {
            return search[item[0]] = checkBool(item[1]);
        });

    return search;
};

export const passportFieldsOfUser = function (user: any, field: string) {
    const passportData = UserInfoHandler.getPassportData.call(user);

    return passportData && passportData[field]
        ? passportData[field] : EMPTY_DATA;
};

export const getRegistrationString = (user: any) => {
    const passportData = UserInfoHandler.getPassportData.call(user);

    return passportData
        && [passportData.registration_region || EMPTY_DATA,
            passportData.registration_locality || EMPTY_DATA,
            passportData.registration_street || EMPTY_DATA,
            passportData.registration_house || EMPTY_DATA,
            passportData.registration_apartment || EMPTY_DATA].join(', ');
};

export const driveLicenseFieldsOfUser = function (user: any, field: string) {
    const driveLicenseData = UserInfoHandler.getDrivingLicenseData.call(user);

    return driveLicenseData && driveLicenseData[field]
        ? driveLicenseData[field] : EMPTY_DATA;
};

export const getUserInfoForCSV = (user: any) => {
    const PASSPORT_SERIES_SPLIT_INDEX = 2;
    const PASSPORT_NUMBER_START_INDEX = 4;

    const lastName = user.last_name || EMPTY_DATA;
    const firstName = user.first_name;
    const patronymicName = user.pn;

    const passportData = UserInfoHandler.getPassportData.call(user);
    const passportSeries = `${passportData?.doc_value?.substring(0, PASSPORT_SERIES_SPLIT_INDEX) || ''}`
        + ` ${passportData?.doc_value?.substring(PASSPORT_SERIES_SPLIT_INDEX, PASSPORT_NUMBER_START_INDEX) || ''}`;
    const passportNumber = passportData?.doc_value?.substring(PASSPORT_NUMBER_START_INDEX) || '';
    const passportIssueDate = getFormatDateString({
        value: passportData.issue_date,
        onlyDate: true,
    });

    const passportRealiser = EMPTY_DATA;
    const passportBirthDate = getFormatDateString({
        value: passportFieldsOfUser(user, 'birth_date'),
        onlyDate: true,
    });
    const passportBirthPlace = passportFieldsOfUser(user, 'birth_place');
    const passportRegistrationPlace = getRegistrationString(user);

    const driverLicenseNumber = driveLicenseFieldsOfUser(user, 'number_front');
    const driverLicenseCategories = driveLicenseFieldsOfUser(user, 'categories');
    const driverLicenseExp = getFormatDateString(
        {
            value: driveLicenseFieldsOfUser(user, 'experience_from') != '—' ?
                driveLicenseFieldsOfUser(user, 'experience_from') :
                driveLicenseFieldsOfUser(user, 'categories_b_valid_from_date'),
            onlyDate: true,
        });
    const driverLicenseFrom = getFormatDateString({
        value: driveLicenseFieldsOfUser(user, 'issue_date'),
        onlyDate: true,
    });
    const driverLicenseValidToDate = getFormatDateString({
        value: driveLicenseFieldsOfUser(user, 'categories_b_valid_to_date'),
        onlyDate: true,
    });

    const phone = user && user.setup && user.setup.phone && user.setup.phone.number;
    const email = user && user.setup && user.setup.email && user.setup.email.address;

    function getFormatDateString(value: any) {
        return FormatDate(value).props.children;
    }

    return `${firstName};${lastName};${patronymicName};${passportSeries};${passportNumber};${passportIssueDate};${passportRealiser};${passportBirthDate};${passportBirthPlace};${passportRegistrationPlace};${driverLicenseNumber};${driverLicenseCategories};${driverLicenseExp};${driverLicenseFrom};${driverLicenseValidToDate};${phone};${email}\n`;

};

export function debounce(f, ms: number) {
    let timer;

    return function (this: any, ...args) {
        clearTimeout(timer);
        timer = setTimeout(() => {
            f.apply(this, args);
        }, ms);
    };
}

export const pad = (number, length) => {
    let result = number.toString();
    while (result.length < length) {
        result = '0' + result;
    }

    return result;
};

export const makeObjPlain = (obj: any, prevKeys = '') => {
    const result: any = {};

    Object.entries(obj).forEach((entry: any) => {
        const [key, value] = entry;
        if (typeof value !== 'object') {
            result[`${prevKeys && `${prevKeys}.`}${key}`] = value;
        } else {
            if (value === null || typeof value === 'undefined') {
                result[`${prevKeys && `${prevKeys}.`}${key}`] = value;
            } else {
                const insertedValue = makeObjPlain(value, key);
                Object.entries(insertedValue).forEach((entry: any) => {
                    const [key, value] = entry;
                    result[`${prevKeys && `${prevKeys}.`}${key}`] = value;
                });
            }
        }
    });

    return result;
};

export const normalizeHash = () => {
    return (location.hash[location.hash.length - 1] !== '/')
        ? (location.hash + '/')
        : location.hash;
};

export const normalizeHash2 = (props: { carId: string; extraPath: string; href: string }) => {
    const { carId, extraPath, href } = props;
    const back = href?.split?.('back=')?.[1];

    return `#/cars/${carId}/tags/${extraPath}${back ? `?back=${back}` : ''}`;
};

export const isValidGUIDString = (value: string) => {
    const regExp = /^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i;

    return regExp.test(value.toLowerCase());
};

export const replaceUserNameToLink = (userName: string) => {
    return isValidGUIDString(userName)
        ? <Link className={'link'}
                title={userName}
                href={`#/clients/${userName}/tags-history`}>[username]</Link> : userName;
};

export const openNewTab = (href: string) => {
    const a: HTMLAnchorElement = document.createElement('a');
    a.target = '_blank';
    a.href = href;
    a.style.display = 'none';
    document.body.appendChild(a);
    a.click();
    a.remove();
};

export const deepCopy = (inObject: Dict<any>): any => {
    let value: any;
    let key: string;

    if (typeof inObject !== 'object' || inObject === null) {
        return inObject; // Return the value if inObject is not an object
    }

    // Create an array or object to hold the values
    const outObject: Dict<any> | any[] = Array.isArray(inObject) ? [] : {};

    for (key in inObject) {
        value = inObject[key];

        // Recursively (deep) copy for nested objects, including arrays
        outObject[key] = (typeof value === 'object' && value !== null) ? deepCopy(value) : value;
    }

    return outObject;
};

export interface IIncident {
    userName: string;
    userPhone: string;
    clientUrl: string;
    carNumber: string;
    autoUrl: string;
    rideStart: number;
    rideUrl: string;
    carLocation: Dict<number>;
    carModel: string;
    city_area: string;
    carVin: string;
    district: string;
    insurance?: string;
    insurance_provider?: string;
}

export const openIncidentForm = (params: IIncident) => {
    let address;
    const {
        userName,
        userPhone,
        clientUrl,
        carNumber,
        autoUrl,
        rideStart,
        rideUrl,
        carLocation,
        carModel,
        carVin,
        city_area,
        district,
        insurance,
        insurance_provider,
    } = params;

    const link = `https://forms.yandex-team.ru/surveys/24340/`
        + `?name=${encodeURIComponent(userName)}`
        + `&phone=${encodeURIComponent(userPhone)}`
        + `&client_url=${encodeURIComponent(clientUrl)}`
        + `&auto_id=${encodeURIComponent(carNumber)}`
        + `&auto_url=${encodeURIComponent(autoUrl)}`
        + `&ride_date=${rideStart ? encodeURIComponent(new Date(rideStart).toLocaleString('ru')) : ''}`
        + `&ride_url=${encodeURIComponent(rideUrl)}`
        + `&lon=${encodeURIComponent(carLocation && carLocation[0] || 0)}`
        + `&lat=${encodeURIComponent(carLocation && carLocation[1] || 0)}`
        + `&auto_model=${encodeURIComponent(carModel)}`
        + `&city_area=${encodeURIComponent(city_area)}`
        + `&car_vin=${encodeURIComponent(carVin)}`
        + `&insurance=${insurance && encodeURIComponent(insurance)}`
        + `&insurance_provider=${insurance_provider && encodeURIComponent(insurance_provider)}`;

    if (carLocation) {
        ymaps.geocode(carLocation)
            .then((res: any) => {
                const firstGeoObject = res.geoObjects.get(0);
                address = firstGeoObject.getAddressLine();
                const gps_info = (carLocation?.[1] + ',' + carLocation?.[0]).trim() || '';

                openNewTab(
                    link + `${address ? `&adress=${encodeURIComponent(address)}` : ''}`
                    + `${gps_info ? `&gps_info=${encodeURIComponent(gps_info)}` : ''}`
                    + `${district ? `&district=${encodeURIComponent(district)}` : ''}`,
                );
            })
            .catch(() => openNewTab(link));
    } else {
        openNewTab(link);
    }
};

//return string with TS-styled interface for objects array
export const getObjectsArrInterface = (items: any[], name = 'I', withOptional = true) => {
    const OPTIONAL_TYPE = 'undefined';
    const DICT_KEY = 'Dict';
    const ANY_KEY = 'any';

    const isObject = (obj: any) => {
        return obj && typeof obj === 'object' && !Array.isArray(obj);
    };

    const getObjectTypes = (items: any[], parents: string[] = []) => {

        let keys: string[] = items?.reduce((result: string[], curr) => {
            result.push(...Object.keys(curr));

            return result;
        }, []) || [];
        keys = Array.from(new Set(keys));

        return keys.reduce((result: Dict<any>, curr) => {
            //get TS types from all items with keys
            let keyTypes = items.map(item => {
                const itemType = typeof item?.[curr];
                let type: string = itemType;

                switch (itemType) {
                case 'function':
                    type = ANY_KEY;
                    break;
                case 'object':
                    type = isObject(item?.[curr]) ? DICT_KEY : `${ANY_KEY}[]`;
                    break;
                case 'symbol':
                    type = ANY_KEY;
                    break;
                }

                return type;
            });
            keyTypes = Array.from(new Set(keyTypes));

            //Marks value as 'any', if primitive and Dict<any>
            const existingKeyTypes = keyTypes.filter(keyType => keyType !== OPTIONAL_TYPE);
            if (existingKeyTypes.includes(DICT_KEY) && existingKeyTypes.length > 1) {
                keyTypes = [...keyTypes.filter(keyType => keyType === OPTIONAL_TYPE), ANY_KEY];
            }

            //get included types, if Dict<any>
            if (keyTypes.includes(DICT_KEY)) {
                const currentItems = items?.map(item => {
                    return [...parents, curr].reduce((result: Dict<any>, parent) => {
                        return item[parent];
                    }, item) ?? {};
                });
                const included = getObjectTypes(currentItems, [...parents, curr]);
                result[curr] = { _interface_types: keyTypes, _object_data: included };
            } else {
                result[curr] = { _interface_types: keyTypes };
            }

            return result;
        }, {});
    };

    const types = getObjectTypes(items);

    //converts types object to TS-styled string
    const convertTypesToString = (types: Dict<any>) => {
        return `{\n`
            + `${Object.entries(types)
                .map(entry => {
                    const [key, types] = entry;

                    if (!withOptional && types._interface_types.includes(OPTIONAL_TYPE)) {
                        return '';
                    }

                    const keyPart = `${key}${types._interface_types.includes(OPTIONAL_TYPE) ? '?' : ''}: `;

                    const typePart = types._object_data
                        ? `${convertTypesToString(types._object_data)},\n`
                        : `${types._interface_types.filter(type => type !== OPTIONAL_TYPE).join(' | ')},\n`;

                    return `\t${keyPart}${typePart}`;
                }).join('')}`
            + `}`;
    };

    return `interface ${name} ${convertTypesToString(types)}`;
};

export const deleteNull = (obj: any) => {
    for (const key in obj) {
        if (isObject(obj[key])) {
            deleteNull(obj[key]);
        } else {
            if (obj[key] === null || obj[key] === '') {
                delete obj[key];
            }
        }
    }
};

export const isObject = (obj: any) => {
    return obj && typeof obj === 'object';
};

export const setDocumentTitle = (data = 'Я.Драйв') => {
    document.title = data;
};

export const formatNumberToStringWithSpaces = (value: number) => {
    const threeNumRegex = /\B(?=(\d{3})+(?!\d))/g;
    const divider = ' ';

    return value?.toString()?.replace(threeNumRegex, divider);
};

export function throttle(func, ms) {
    let isThrottled = false,
        savedArgs: any,
        savedThis: any;

    function wrapper(this: any) {
        if (isThrottled) {
            savedArgs = arguments;
            savedThis = this;

            return;
        }

        func.apply(this, arguments);

        isThrottled = true;

        setTimeout(function () {
            isThrottled = false;
            if (savedArgs) {
                wrapper.apply(savedThis, savedArgs);
                savedArgs = savedThis = null;
            }
        }, ms);
    }

    return wrapper;
}

export const getDuration = (from: number) => {
    const _from = from * ONE_SECOND || new Date().getTime();

    return (new Date().getTime() - _from) / ONE_SECOND;
};

export const isDateValid = (date: number | string) => {
    return (new Date(date))?.getTime() > 0;
};

export const makeLatCarNumber = (carNumber: string): string => {
    const ruLat = {
        'а': 'a',
        'в': 'b',
        'е': 'e',
        'к': 'k',
        'м': 'm',
        'н': 'h',
        'о': 'o',
        'р': 'p',
        'с': 'c',
        'т': 't',
        'у': 'y',
        'х': 'x',
    };

    return carNumber
        .split('')
        .map((sym) => ruLat?.[sym?.toLowerCase()] ?? sym)
        .join('');
};

export const makeKirCarNumber = (carNumber: string): string => {
    const latRu = {
        'a': 'а',
        'b': 'в',
        'e': 'е',
        'k': 'к',
        'm': 'м',
        'h': 'н',
        'o': 'о',
        'p': 'р',
        'c': 'с',
        't': 'т',
        'y': 'у',
        'x': 'х',
    };

    return carNumber
        .split('')
        .map((sym) => latRu?.[sym?.toLowerCase()] ?? sym)
        .join('');
};

export const truncateString = (string, maxLength) => {
    return (string.length > maxLength) ?
        string.slice(0, maxLength - 1) + '…' : string;
};

export const isTest = (): boolean => {
    return process.env.NODE_ENV === RUNNING_MODE.TEST;
};
