import produce from 'immer';
import * as React from 'react';

import { GLOBAL_SEARCH_OBJECTS } from '../../core/ui/IDSelect/constants';
import { Request2 } from '../../core/utils/request';
import { isValidGUIDString } from '../../core/utils/utils';
import { REQUESTS, SEARCH_REQUESTS } from './request';
import { ISearchData, ISearchOptions } from './types';

const MIN_CHARS_FOR_SEARCH = 2;
const DEFAULT_RESPONSE_LIMITS = 100;

const maybePhoneFunc = (text) => {
    return text.trim().replace(/[\+|\-|{\s}+]/ig, '');
};

export const GlobalSearchProvider = () => (Component: any): any => {
    return class extends React.Component {
        state = {
            isLoading: false,
            error: null,
            data: {} as ISearchData,
            isEmpty: true,
        };
        timer: any;
        request = new Request2({ requestConfigs: SEARCH_REQUESTS });

        findTextPrepare(text) {
            const maybePhone = maybePhoneFunc(text);
            if (!isNaN(maybePhone)) {
                return maybePhone;
            }

            const searchText = text
                .trim()
                .split(' ')
                .map(item => {
                    const maybePhone = maybePhoneFunc(text);
                    if (!isNaN(maybePhone)) {
                        return maybePhone;
                    }

                    return item;
                })
                .join(',');

            return encodeURIComponent(searchText);
        }

        extraSearch(has_all_of) {
            return this.request.exec(REQUESTS.GET_USER, { queryParams: { user_id: has_all_of } })
                .then(response => {
                    const obj = {
                        objects: {
                            users: [response],
                        },
                    };

                    return obj;
                });
        }

        search(options: ISearchOptions) {
            const { has_all_of, limit = DEFAULT_RESPONSE_LIMITS, object, withSessions } = options;
            const TIMEOUT = 300;
            const detectSessionsTextCount = 30;
            clearTimeout(this.timer);
            this.timer = has_all_of
                && has_all_of.length >= MIN_CHARS_FOR_SEARCH
                && setTimeout(() => {
                    this.setState(produce(this.state, draft => {
                        draft.isLoading = true;
                        draft.error = null;
                    }), () => {
                        this.request.abort();

                        const what = object !== GLOBAL_SEARCH_OBJECTS.all ? object : null;
                        Promise.allSettled([this.request.exec(REQUESTS.SEARCH, {
                            queryParams: {
                                has_all_of: this.findTextPrepare(has_all_of),
                                limit,
                                what,
                            },
                        }), withSessions && has_all_of.length > detectSessionsTextCount
                            ? this.request.exec(REQUESTS.GET_SESSION, { queryParams: { session_id: has_all_of } })
                            : Promise.resolve(null)])
                            .then(async (responses: any[]) => {
                                if (responses[0].status === 'rejected') {
                                    this.setState(produce(this.state, draft => {
                                        draft.isLoading = false;
                                        draft.error = responses[0].value;
                                        draft.isEmpty = true;
                                    }));
                                }

                                const response = responses[0]?.value;
                                const session = responses[1]?.value;

                                response.mixedSession = session;

                                let extraResponse: any = {};

                                if (isValidGUIDString(has_all_of)) {
                                    try {
                                        extraResponse = await this.extraSearch(has_all_of);
                                    } catch (err) {

                                    }
                                }

                                if (extraResponse?.objects?.users?.length) {
                                    response.objects.users.splice(0, 0, extraResponse.objects.users[0]);
                                }

                                this.setState(produce(this.state, draft => {
                                    draft.isLoading = false;
                                    draft.data = response;
                                    draft.error = null;
                                    draft.isEmpty = !(response && response.objects
                                        && (
                                            response.objects.cars
                                            && response.objects.cars.length &&
                                            response.objects.users
                                            && response.objects.users.length));
                                }));
                            })
                            .catch(error => {
                                this.setState(produce(this.state, draft => {
                                    draft.isLoading = false;
                                    draft.error = error;
                                    draft.isEmpty = true;
                                }));
                            });
                    });
                }, TIMEOUT);
        }

        componentWillUnmount(): void {
            clearTimeout(this.timer);
        }

        render() {
            return <Component {...this.props}
                              searchIsLoading={this.state.isLoading}
                              searchError={this.state.error}
                              searchIsEmpty={this.state.isEmpty}
                              search={this.search.bind(this)}
                              searchData={this.state.data}/>;
        }
    };
};
