import * as React from 'react';

import { ISchemaItem } from '../../core/components/FormConstructor/types';
import { ITableProps } from '../../core/components/InterfaceAdminConfig/CommonIntefaceConfig';
import { HOST } from '../../core/components/Migrate/constants';
import { getActuality } from '../../core/constants';
import { Confirm } from '../../core/ui/FullModal';
import { isObjectEqual } from '../../core/utils/isObjectEqual';
import { isValidJSONString } from '../../core/utils/isValidJSONString';
import LS from '../../core/utils/localStorage/localStorage';
import { Request2 } from '../../core/utils/request';
import { deepCopy } from '../../core/utils/utils';
import EditFormModal from './EditFormModal/component';
import EditRawFormModal from './EditRawFormModal/component';
import { INTERFACE_ADMIN_REQUESTS, REQUESTS } from './request';

interface IInterfaceAdminProviderState {
    waitingForResponse: boolean;
    isDataLoading: boolean;
    isEditModalOpen: boolean;
    isEditRawModalOpen: boolean;
    isDeleteConfirmOpen: boolean;
    isAddingAction: boolean;
    modalError: Error | null;
    rawModalError: Error | null;
    isRawLoading: boolean;
    error: Error | null;
    data: Record<string,any>;
    isDataChanged: boolean;

    changedData: Record<string,any>;
    isFormValid: boolean;
}

export interface IInterfaceAdminProviderProps {
    isDataReady?: boolean;
    isLoading?: boolean;
    error?: Error | null;
    data?: Record<string,any> [];
    getDataByKey?: (key: string, schema: Record<string,any>) => Promise<Record<string,any> []>;
    editData?: (item: Record<string, any>) => void;
    addValue?: () => void;
    editRawData?: () => void;
    deleteValue?: (item: Record<string,any>) => void;
    setIsDataChanged?: (isDataChanged: boolean) => void;
}

export const InterfaceAdminProvider = () => (Component): any => {
    return class extends React.Component<ITableProps, IInterfaceAdminProviderState> {
        state: IInterfaceAdminProviderState = {
            waitingForResponse: false,
            isDataLoading: false,
            isEditModalOpen: false,
            isEditRawModalOpen: false,
            isDeleteConfirmOpen: false,
            isAddingAction: false,
            modalError: null,
            rawModalError: null,
            error: null,
            data: [],
            isDataChanged: false,

            changedData: {},
            isFormValid: true,
            isRawLoading: false,

        };
        request = new Request2({ requestConfigs: INTERFACE_ADMIN_REQUESTS });
        settingKey: string;
        initialData: Record<string,any>;
        formSchema: Record<string,ISchemaItem>;
        dataBeforeChanged: Record<string,any> [];
        ls = new LS();

        componentWillUnmount() {
            this.request.abort();
        }

        getDataByKey(tabKey: string, formSchema: Record<string,any>) {
            this.settingKey = tabKey;
            this.formSchema = formSchema;

            this.setState({
                isDataLoading: true,
                error: null,
            }, () => {
                this.request.exec(REQUESTS.GET_GLOBAL_SETTINGS, {
                    queryParams: {
                        prefix: this.settingKey,
                        actuality: getActuality(),
                    },
                    ...this.ls.envIsProdPre() && {
                        headers: {
                            'saasnew': HOST.ADMIN,
                        },
                    },
                })
                    .then((request) => {
                        const stringData = request?.settings?.[0]?.setting_value ?? [];
                        const data = isValidJSONString(stringData) ? JSON.parse(stringData) : stringData;

                        this.setState({
                            data,
                            isDataLoading: false,
                        });
                    })
                    .catch(error =>
                        this.setState({
                            isDataLoading: false,
                            error,
                        }));
            });
        }

        onEditData(data: Record<string,any>) {
            this.initialData = deepCopy(data);
            this.editModalOpen(true);
        }

        onEditRawData() {
            this.editRawModalOpen(true);
        }

        onAddValue() {
            this.onEditData({});
        }

        onDeleteValue(item: Record<string,any>) {
            this.initialData = item;
            this.openDeleteConfirm(true);
        }

        prepareData(data) {
            if (this._checkIfUniqueItem()) {
                data.push(this.state.changedData);
                this._saveData(data);
            } else {
                this.setState({ modalError: new Error('Такой элемент уже существует') });
            }
        }

        onEditDataSave() {
            this.dataBeforeChanged = this.state.data.slice();

            const data = this._siftData();
            this.prepareData(data);
        }

        onAddDataSave() {
            const { data } = this.state;
            this.dataBeforeChanged = data.slice();

            this.prepareData(data);
        }

        onDeleteItemConfirm() {
            this.dataBeforeChanged = this.state.data.slice();

            const data = this._siftData();
            this._saveData(data);
        }

        _checkIfUniqueItem() {
            const { data, changedData } = this.state;

            return data.every(el => !isObjectEqual(el, changedData));
        }

        _siftData(): Record<string,any> {
            const { data: prevData, isDataChanged } = this.state;

            const excessItemIndex = isDataChanged
                ? prevData.findIndex(el => el.id === this.initialData.id && el.name === this.initialData.name)
                : prevData.findIndex(el => isObjectEqual(el, this.initialData)) ?? {};

            if (excessItemIndex !== -1) {
                prevData.splice(excessItemIndex, 1);
            }

            return prevData;
        }

        _saveData(newSettingValues) {
            let error;
            const { validator } = this.props;
            const { changedData } = this.state;

            if (validator) {
                error = validator(newSettingValues, changedData);
            }

            if (error) {
                this.setState({
                    data: this.dataBeforeChanged,
                    modalError: error,
                    waitingForResponse: false,
                    isRawLoading: false,
                });
            } else {
                this.setState({
                    modalError: null,
                    waitingForResponse: true,
                    rawModalError: null,
                    isRawLoading: true,
                }, () => {
                    const prop = (host?: string) => {
                        let setting_value;

                        try {
                            setting_value = JSON.stringify(newSettingValues);
                        } catch {
                            setting_value = newSettingValues;
                        }

                        return {
                            body: {
                                settings: [
                                    {
                                        setting_key: this.settingKey,
                                        setting_value,
                                    },
                                ],
                            },
                            ...host && {
                                headers: {
                                    'saasnew': host,
                                },
                            },
                        };
                    };

                    let settingRequest;

                    if (this.ls.envIsProdPre()) {
                        settingRequest = Promise.all([
                            this.request.exec(REQUESTS.UPSERT_SETTINGS, prop(HOST.ADMIN)),
                            this.request.exec(REQUESTS.UPSERT_SETTINGS, prop(HOST.PRESTABLE)),
                        ]);
                    } else {
                        settingRequest = this.request.exec(REQUESTS.UPSERT_SETTINGS, prop());
                    }

                    settingRequest && settingRequest.then(() => {
                        this.setState({
                            data: newSettingValues,
                            isEditModalOpen: false,
                            isDeleteConfirmOpen: false,
                            waitingForResponse: false,
                            isRawLoading: false,
                            isEditRawModalOpen: false,
                        });
                    })
                        .catch(error => {
                            this.setState({
                                data: this.dataBeforeChanged,
                                modalError: error,
                                waitingForResponse: false,
                                rawModalError: error,
                                isRawLoading: false,
                            });
                        });
                });
            }
        }

        openDeleteConfirm(isOpen: boolean) {
            this.setState({
                isDeleteConfirmOpen: isOpen,
                modalError: null,
            });
        }

        editModalOpen(isOpen: boolean) {
            this.setState({
                isEditModalOpen: isOpen,
                modalError: null,
            });
        }

        editRawModalOpen(isOpen: boolean) {
            this.setState({
                isEditRawModalOpen: isOpen,
                rawModalError: null,
            });
        }

        onModalDataChange(newData, isFormValid: boolean) {
            this.setState({
                changedData: newData,
                isFormValid,
            });
        }

        onEditRawDataSave(rawData: string) {
            this._saveData(rawData);
        }

        setIsDataChanged(isDataChanged: boolean) {
            this.setState({ isDataChanged });
        }

        render() {
            const {
                error,
                isRawLoading,
                isDataLoading,
                isEditModalOpen,
                isEditRawModalOpen,
                changedData,
                isAddingAction,
                waitingForResponse,
                isDeleteConfirmOpen,
                isFormValid,
                modalError,
                rawModalError,
                data,
                isDataChanged,
            } = this.state;

            return <>
                <Component {...this.props}
                           isLoading={isDataLoading}
                           error={error}
                           data={data}
                           getDataByKey={this.getDataByKey.bind(this)}
                           editData={this.onEditData.bind(this)}
                           addValue={this.onAddValue.bind(this)}
                           editRawData={this.onEditRawData.bind(this)}
                           deleteValue={this.onDeleteValue.bind(this)}
                           setIsDataChanged={this.setIsDataChanged.bind(this)}/>

                {isEditModalOpen
                    ? <EditFormModal error={modalError}
                                     changedData={changedData}
                                     isFormValid={isFormValid}
                                     isAddingAction={isAddingAction}
                                     isDataChanged={isDataChanged}
                                     waitingForResponse={waitingForResponse}
                                     formSchema={this.formSchema}
                                     initialData={this.initialData}
                                     onModalDataChange={this.onModalDataChange.bind(this)}
                                     onEditDataSave={this.onEditDataSave.bind(this)}
                                     onAddDataSave={this.onAddDataSave.bind(this)}
                                     editModalOpen={this.editModalOpen.bind(this)}/>
                    : null
                }
                {isEditRawModalOpen
                    ? <EditRawFormModal error={rawModalError}
                                        initialData={data}
                                        isRawLoading={isRawLoading}
                                        onEditRawDataSave={this.onEditRawDataSave.bind(this)}
                                        editRawModalOpen={this.editRawModalOpen.bind(this)}/>
                    : null
                }

                {isDeleteConfirmOpen
                    ? <Confirm error={modalError}
                               onClose={this.openDeleteConfirm.bind(this, false)}
                               accept={this.onDeleteItemConfirm.bind(this)}
                               isWorking={waitingForResponse}
                               question={'Вы действительно хотите удалить элемент?'}/>
                    : null
                }
            </>;
        }
    };
};
