import produce from 'immer';
import * as React from 'react';
import { connect } from 'react-redux';

import { ONE_SECOND } from '../../core/constants';
import { FormatDateInString } from '../../core/ui/FormatDate';
import { IOptionInfo } from '../../core/ui/Select';
import { ConstantsKey, fetchConstants } from '../../core/utils/fetchConstants';
import { isValidJSONString } from '../../core/utils/isValidJSONString';
import { Request2 } from '../../core/utils/request';
import { Dict } from '../../types';
import { REQUESTS, TAG_HISTORY_REQUESTS } from './request';

const arrToObj = (arr: any[]) => {
    return arr.reduce((_p, _c) => {
        if (!_p.hasOwnProperty(_c.name)) {
            const _meta = isValidJSONString(_c.meta)
                ? JSON.parse(_c.meta)
                : typeof(_c.meta) === 'object' ? _c.meta : {};
            _p[_c.name] = Object.assign({}, _c, { _meta });
        }

        return _p;
    }, {});
};

const TAG_DESCRIPTION_RES = 2;
const mapStateToProps = () => {
    return {};
};

const mapDispatchToProps: any = (dispatch) => {
    return {
        constants: () => fetchConstants([ConstantsKey.IFACE_TAG_FILTERS], dispatch),
    };
};

export const TagHistoryWith = () => (Component): any => {
    class Wrapper extends React.Component<any, any> {
        state = {
            isLoading: false,
            error: null,
            data: {} as ITagHistoryData,
        };
        request = new Request2({ requestConfigs: TAG_HISTORY_REQUESTS });

        getTagsHistory(options: { object_id: string; duration?: string }) {
            this.setState(produce(this.state, draft => {
                draft.isLoading = true;
                draft.error = null;
            }), () => {

                Promise.all([
                    this.request.exec(REQUESTS.GET_TAG_HISTORY, {
                        queryParams: {
                            object_id: options.object_id,
                            duration: options.duration || '90d',
                        },
                    }),
                    this.props?.constants?.(),
                    this.request.exec(REQUESTS.GET_TAG_DESCRIPTIONS),
                ])
                    .then(response => {
                        const records = arrToObj(response[TAG_DESCRIPTION_RES].records);

                        const bonuses = Object.assign({}, response[0], {
                            records: response[0]?.records
                                ?.filter(r => response[1]?.iface_tag_filters?.bonus?.includes(r?.tag_name))
                                || [],
                        });
                        const bonusesMeta = response[TAG_DESCRIPTION_RES]?.records?.reduce((_p, _c) => {
                            if (response[1]?.iface_tag_filters?.bonus?.includes(_c?.name)) {
                                let meta = _c.meta;
                                if (meta) {
                                    meta = isValidJSONString(meta)
                                        ? JSON.parse(meta)
                                        : typeof(meta) === 'object' ? meta : {};
                                }

                                _p = [..._p, {
                                    ...meta,
                                    name: _c.name,
                                }];
                            }

                            return _p;
                        }, []);
                        const bonusesOptions = response[TAG_DESCRIPTION_RES]?.records?.reduce((_p, _c) => {
                            if (response[1]?.iface_tag_filters?.bonus?.includes(_c.name)) {
                                const bonusMeta = bonusesMeta?.find((bonus) => bonus?.name === _c.name);
                                _p = [..._p, {
                                    value: _c.name,
                                    text: _c.display_name,
                                    description: `${records?.[_c.name]?.comment}`
                                        + `${bonusMeta?.amount_lifetime
                                            ? `, срок жизни ${bonusMeta?.amount_lifetime}`
                                            : ''}`
                                        + `${bonusMeta?.amount_deadline
                                            ? `, до ${FormatDateInString({
                                                value: bonusMeta?.amount_deadline * ONE_SECOND,
                                            })}`
                                            : ''}`,
                                }];
                            }

                            return _p;
                        }, []);

                        this.setState({
                            isLoading: false,
                            data: {
                                _history: response[0],
                                _constants: response[1],
                                _descriptions: response[TAG_DESCRIPTION_RES],
                                descriptions: {
                                    deprecated: arrToObj(response[TAG_DESCRIPTION_RES].deprecated),
                                    records,
                                },
                                bonuses,
                                bonusesOptions,
                                bonusesMeta,
                            },
                            error: null,
                        });
                    })
                    .catch(error => {
                        this.setState({
                            isLoading: false,
                            error,
                        });
                    });
            });
        }

        render() {
            return <Component {...this.props}
                              tagHistoryIsLoading={this.state.isLoading}
                              tagHistoryError={this.state.error}
                              getTagsHistory={this.getTagsHistory.bind(this)}
                              tagHistoryData={this.state.data}/>;
        }
    }

    return connect(mapStateToProps, mapDispatchToProps)(Wrapper);
};

export interface ITagHistoryWithPropsExport {
    tagHistoryIsLoading?: boolean;
    tagHistoryError?: any;
    getTagsHistory?: (options: { object_id: string }) => void;
    tagHistoryData?: ITagHistoryData;
}

interface ITagHistoryData {
    _history: any;
    _constants: any;
    _descriptions: any;
    descriptions: {
        deprecated: Dict<any>;
        records: Dict<any>;
    };
    bonuses: { objects: Dict<any>; records: []; users: Dict<any> };
    bonusesOptions: IOptionInfo[];
    bonusesMeta?: any;
}
