const path = require('path');
const webpack = require('webpack');
const HtmlWebpackPlugin = require('html-webpack-plugin');
const MiniCssExtractPlugin = require("mini-css-extract-plugin");
const ForkTsCheckerWebpackPlugin = require('fork-ts-checker-webpack-plugin');
const cacheGroups = require('./webpack.cache_groups');

const rootDir = './';
const tsconfig = path.resolve(rootDir, 'tsconfig.json');

module.exports = {
    entry: {
        main: path.resolve(rootDir, 'src/components/App/index.tsx')
    },
    output: {
        path: path.resolve(rootDir, 'dist'),
        chunkFilename: '[name].[contenthash].js',
        filename: '[name].[contenthash].js',
        publicPath: ''
    },
    resolve: {
        extensions: [".ts", ".tsx", ".js"],
        alias: {},
    },
    plugins: [
        new ForkTsCheckerWebpackPlugin({
            async: false,
            issue: {
                scope: 'all',
            },
            typescript: {
                enabled: true,
                memoryLimit: 4096,
                configFile: tsconfig,
            }
        }),
        new MiniCssExtractPlugin({
            filename: '[name].[contenthash].css',
            chunkFilename: '[name].[contenthash].css',
            ignoreOrder: true,
        }),
        new HtmlWebpackPlugin({
            template: path.resolve(rootDir, './src/index.html'),
            filename: 'index.html',
            favicon: path.resolve(rootDir, './src/favicon.svg')
        }),
        new webpack.DefinePlugin({
            'process.env': {
                'VERSION': JSON.stringify(process.env.VERSION),
                'RUNNING_MODE': JSON.stringify(process.env.RUNNING_MODE),
                'BACKEND_CLUSTER': JSON.stringify(process.env.BACKEND_CLUSTER)
            }
        })
    ],
    module: {
        rules: [
            {
                test: /\.tsx?$/,
                exclude: /node_modules/,
                use: [
                    {
                        loader: `ts-loader`,
                        options: {
                            transpileOnly: true,
                            configFile: tsconfig,
                        }
                    }
                ]
            },
            {
                test: /images(.)+\.svg$/,
                use: [
                    {
                        loader: 'url-loader',
                        options: {
                            limit: 8192
                        }
                    }
                ]
            },
            {
                test: /\.(png|jpg|gif)$/,
                use: [
                    {
                        loader: 'url-loader',
                        options: {
                            limit: 0
                        }
                    }
                ]
            },
            {
                test: /inline-svg(.)+\.svg$/,
                use: 'svg-inline-loader',
            },
            {
                test: /\.css$/,
                use: [
                    {
                        loader: MiniCssExtractPlugin.loader
                    },
                    {
                        loader: 'css-loader',
                        options: {
                            modules: {
                                localIdentName: '[folder]__[local]___[hash:base64:5]',
                            },
                            importLoaders: 1
                        },
                    },
                    {
                        loader: 'postcss-loader',
                        options: {
                            postcssOptions: {
                                path: path.resolve(rootDir),
                            }
                        }
                    }
                ]
            },
            {
                test: /fonts(.)+\.(eot|woff2|woff|ttf|svg)$/,
                use: 'file-loader?name=fonts/[name].[ext]'
            },
        ]
    },
    optimization: {
        splitChunks: {
            minSize: 10240,
            cacheGroups,
        },
        noEmitOnErrors: true,
    },
    stats: {
        assetsSort: '!size',
        colors: true,
        logging: 'error',
        reasons: false,
        children: false,
        usedExports: false
    }
};
