import {
    Button,
    createStyles,
    Grid,
    makeStyles,
    Paper,
    Theme,
} from '@material-ui/core';
import { Skeleton } from '@material-ui/lab';
import React, { useEffect } from 'react';
import useStyles from './styles';
import CarDialog from './CarDialog';
import { ICarInfo } from './index';
import useData from '../../hooks/useData';

const COMMANDS_DISPLAY_NAMES = {
    OPEN_DOORS: 'Открыть двери',
    CLOSE_DOORS: 'Закрыть двери',
    YADRIVE_UNLOCK_HOOD: 'Открыть капот',
    YADRIVE_LOCK_HOOD: 'Закрыть капот',
    HORN: 'Гудок',
    SCENARIO_UNLOCK_DOORS_AND_HOOD: 'Открыть капот+двери',
    SCENARIO_LOCK_DOORS_AND_HOOD: 'Закрыть капот+двери',
    SCENARIO_BLE_RESET_FORCED_END_OF_LEASE: 'Заглушить (force)',
    SCENARIO_POLITE_FORCED_END_OF_LEASE: 'Заглушить'
}

const _useStyles = makeStyles((theme: Theme) =>
    createStyles({
        avatar: {
            width: theme.spacing(17),
            height: theme.spacing(7),
        },
        listItem: {
            padding: 0
        },
        skeleton: {
            height: 190
        },
        button: {
            justifyContent: 'start',
            textAlign: 'left',
            overflow: 'hidden',
            textOverflow: 'ellipsis'
        }
    }),
);

interface IMainCarCardProps {
    carInfo: ICarInfo | null,
}

export default function CarCommandsCard({carInfo}: IMainCarCardProps) {
    const [selectedCommand, setSelectedCommand] = React.useState<string | null>(null);
    const [{
        isLoading: isLoadingCommands,
        data: commands
    }, doCommandsFetch] = useData<{ allowed_commands: string[] }>(true);
    const [{isLoading: isRunningCommand, error: runningCommandError}, runCommandFetch] = useData<ICarInfo>(false);

    const mainClasses = useStyles();
    const classes = _useStyles();


    useEffect(() => {
        doCommandsFetch({url: 'api/staff/commands'});
    }, []);
    let allowedCommands = commands?.allowed_commands ?? [];

    const openModal = (command: string | null) => {
        setSelectedCommand(command);
    };

    const onRunCommandConfirm = () => {
        let {imei} = carInfo ?? {};

        runCommandFetch({
            url: `api/staff/car/control`,
            options:
                {
                    method: 'POST',
                    body: JSON.stringify({imei, command: selectedCommand})
                }
        })
    };

    return <>
        <Paper className={mainClasses.card}>
            {isLoadingCommands
                ? <Grid item xs={12} container direction={'column'} justify={'space-around'} className={classes.skeleton}>
                    <Skeleton variant={'rect'} animation={'pulse'} height={60}/>
                    <Skeleton variant={'rect'} animation={'pulse'} height={40}/>
                    <Skeleton variant={'rect'} animation={'pulse'} height={30}/>
                </Grid>
                : <Grid container spacing={1} direction={'row'}>
                    {
                        allowedCommands.map(allowedCommand => {
                            return <Grid key={allowedCommand} item container xs={6}>
                                <Button fullWidth variant={'outlined'} color="secondary"
                                        className={classes.button} onClick={openModal.bind(null, allowedCommand)}>
                                    {COMMANDS_DISPLAY_NAMES?.[allowedCommand] ?? allowedCommand}
                                </Button>
                            </Grid>
                        })
                    }
                </Grid>}
        </Paper>
        <CarDialog title={`Выполнить команду ${selectedCommand}?`}
                   isLoading={isRunningCommand}
                   error={runningCommandError}
                   onConfirm={onRunCommandConfirm}
                   open={!!selectedCommand}
                   openModal={openModal.bind(null, null)}
                   data={carInfo}/>
    </>;
}

