import React, { useEffect } from 'react';
import { useParams } from 'react-router-dom';
import { Container, Grid } from '@material-ui/core';
import useData from '../../hooks/useData';
import ErrorCard from '../../components/ErrorCard';
import MainCarCard from './MainCarCard';
import Doors from './Doors';
import MapCarCard from './MapCarCard';
import CarCommandsCard from './CarCommandsCard';

export interface ISensor {
    timestamp: number,
    name: string,
    value: number | null,
    since: number,
    id: number
}

export interface ICarInfo {
    locations: {
        timestamp: number,
        longitude: number,
        latitude: number,
        course: number,
        type: string,
        since: number,
        precision: number
    }[],
    sensors: ISensor[],
    number: string,
    imei: string,
    id: string,
}

export default function Cars() {
    let {car_number: carNumber, prefix} = useParams<{ car_number: string, prefix?: string }>();
    let [{isLoading, data: carInfo, error}, doFetch] = useData<ICarInfo>(true);

    useEffect(() => {
        if (carNumber) {
            doFetch({
                url: `api/staff/car/info`, options:
                    {
                        method: 'POST',
                        headers: {'Content-Type': 'application/json; charset=utf-8'},
                        body: JSON.stringify({number: carNumber})
                    }
            });
        }
    }, [carNumber]);

    const all = 12;
    const half = 6;
    return <Container maxWidth={'md'}>
        {
            error
                ? <ErrorCard error={error}/>
                : <Grid container spacing={1}>
                    <Grid item xs={all} sm={half}>
                        <MainCarCard find_text={prefix ?? ''}
                                     isLoading={isLoading}
                                     carInfo={carInfo}/>
                    </Grid>
                    <Grid item xs={all} sm={half}>
                        <Doors data={carInfo}/>
                    </Grid>
                    <Grid item xs={all} sm={half}>
                        <CarCommandsCard carInfo={carInfo}/>
                    </Grid>
                    <Grid item xs={all} sm={half}>
                        <MapCarCard carInfo={carInfo}/>
                    </Grid>
                </Grid>
        }
    </Container>;
}

export const getSensorValue = (sensors: ISensor[], name: string) => {
    return sensors.find(sensor => sensor.name === name)?.value ?? null;
};