const fs = require('fs');
const path = require('path');
const ReleaseType = require('./enum/releaseType');
const DeployTarget = require('./enum/deployTarget');

const RELEASE_FOLDER = 'releases/';

const findFolderFromWorkingFolder = (basename) => {
    let folderPath = process.cwd();
    let folderDescriptor;

    while (true) {
        if (folderPath === '/') {
            throw new Error(`Folder ${basename} was not found`);
        }

        try {
            folderDescriptor = fs.opendirSync(path.join(folderPath, basename));
            folderPath = folderDescriptor.path;
            folderDescriptor.closeSync();

            return folderPath;
        } catch (e) {
            folderPath = path.resolve(folderPath, '..');
        }
    }
}

/**
 *  @param {ReleaseType} releaseType
 *  @param {DeployTarget} deployTarget
 *  @returns string
 *  @throws Will throw, if release folder is empty
 */
const getRelease = (releaseType, deployTarget) => {
    if (releaseType === ReleaseType.LATEST && deployTarget === DeployTarget.STAND) {
        return 'dist';
    }

    let releaseFolderPath = findFolderFromWorkingFolder(RELEASE_FOLDER);
    let releaseFolders = releaseFolderPath
        ? fs.readdirSync(releaseFolderPath, {withFileTypes: true})
        : [];

    let releaseFoldersStats = releaseFolders
        .reduce((acc, _folder) => {
            if (_folder.isDirectory()) {
                let folderPath = path.join(releaseFolderPath, _folder.name);
                let folderStats = fs.statSync(folderPath);
                let folderFiles = fs.readdirSync(folderPath);
                let isEmpty = true;

                if (folderFiles && folderFiles.length) {
                    isEmpty = false;
                }

                acc.push({
                    _path: _folder.name,
                    mtime: folderStats.mtime,
                    isEmpty
                });
            }

            return acc;
        }, [])
        .sort((a, b) => b.mtime - a.mtime);

    let releaseVersion = releaseFoldersStats[deployTarget === DeployTarget.PRODUCTION && releaseType === ReleaseType.PREVIOUS ? 1 : 0];

    if (releaseVersion.isEmpty) {
        throw new Error(`Release folder ${releaseVersion._path} is empty`);
    } else {
        return `${RELEASE_FOLDER}${releaseVersion._path}`;
    }
}

module.exports = {
    getRelease
}

