const utils = require('@yandex-int/qtools/lib/utils');
const deployProvider = require('@yandex-int/qtools/lib/providers/deploy');
const {awacs} = require('@yandex-int/awacs-api');
const awacsProvider = require('@yandex-int/qtools/lib/providers/awacs');
const yaml = require('js-yaml');
const fs = require('fs');
const path = require('path');

const DEPLOY_UNIT = 'front';
const AUTH = {
    type: awacs.model.Auth.AuthType.STAFF,
    staff: {
        owners: {
            logins: [],
            groupIds: []
        }
    }
};

async function createBackend({namespaceId, id, stageConfig}) {
    const deployUnit = stageConfig.spec.deploy_units[DEPLOY_UNIT];
    const clusters = deployUnit.replica_set ?
        Object.keys(deployUnit.replica_set.per_cluster_settings) :
        deployUnit.multi_cluster_replica_set.replica_set.clusters.map(({cluster}) => cluster);

    const spec = {
        selector: {
            type: awacs.model.BackendSelector.Type.YP_ENDPOINT_SETS_SD,
            allowEmptyYpEndpointSets: false,
            port: {
                policy: awacs.model.Port.Policy.KEEP
            },
            ypEndpointSets: clusters.map((cluster) => ({
                cluster,
                endpointSetId: `${stageConfig.meta.id}.${DEPLOY_UNIT}`,
                port: {
                    policy: awacs.model.Port.Policy.KEEP
                },
                weight: {
                    policy: awacs.model.Weight.Policy.KEEP
                }
            }))
        }
    };

    try {
        const {backend} = await awacsProvider.getBackend({
            namespaceId,
            id
        });
        if (backend) {
            const currentClusters = backend.spec.selector.ypEndpointSets.map(({cluster}) => cluster);
            if (currentClusters.sort().join() === clusters.sort().join()) {
                return;
            }

            await awacsProvider.updateBackend({
                meta: {
                    version: backend.meta && backend.meta.version || null,
                    id,
                    namespaceId,
                    comment: `Update backend ${id}`
                },
                spec
            });
            //logger.debug(`Backend "${id}" will be updated: https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/${namespaceId}/backends/list/${id}/show/`);
            return;
        }
    } catch (e) {}

    await awacsProvider.createBackend({
        meta: {
            id,
            namespaceId,
            comment: `Create backend ${id}`,
            auth: AUTH
        },
        spec
    });
    //logger.debug(`Backend "${id}" will be created: https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/${namespaceId}/backends/list/${id}/show/`);
}

async function updateUpstream({namespaceId, id, upstreamYaml}) {
    const {upstream} = await awacsProvider.getUpstream({
        id,
        namespaceId
    }).catch(() => ({}));

    const version = upstream ? upstream.meta.version : null;
    const upstreamConfig = {
        meta: {
            id,
            namespaceId,
            comment: `Update/create upstream ${id}`,
            auth: AUTH,
            version
        },
        spec: {
            labels: {
                order: '10000000'
            },
            type: 'YANDEX_BALANCER',
            yandexBalancer: {
                mode: awacs.model.YandexBalancerUpstreamSpec.ConfigMode.EASY_MODE2,
                // Не используем `config`, т.к. хочется сравнивать, а в `config` много лишнего.
                yaml: yaml.dump(upstreamYaml)
            }
        }
    };

    if (!version) {
        await awacsProvider.createUpstream(upstreamConfig);
        //logger.verbose(`Upstream "${id}" will be created.`);
    } else if (upstream.spec.yandexBalancer.yaml !== upstreamConfig.spec.yandexBalancer.yaml) {
        await awacsProvider.updateUpstream(upstreamConfig);
        //logger.verbose(`Upstream "${id}" will be updated.`);
    } else {
        //logger.verbose(`Upstream "${id}" is up to date.`);
    }
    //logger.debug(`Upstream "${id}": https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/${namespaceId}/upstreams/list/${id}/show/`);
}

module.exports = async (ticket) => {
    const qtoolsConfig = utils.getQtoolsConfig(null, {yd: true});
    if (!qtoolsConfig) {
        throw new Error('Cannot get qtools config');
    }

    const {environments} = qtoolsConfig.deploy;
    const environmentConfig = environments.stand;
    const stageConfig = deployProvider.getStageConfig(environmentConfig);
    const {namespaceId, upstreamConfig} = environmentConfig.balancer;

    try {
        // Create backend, if it does not exist.
        await createBackend({namespaceId, id: ticket, stageConfig});

        //Find, compare and update/create upstream.
        await updateUpstream({
            namespaceId,
            id: ticket,
            upstreamYaml: yaml.load(fs.readFileSync(path.resolve(process.cwd(), upstreamConfig)), {skipInvalid: true})
        });
    } catch (error) {
        //logger.error(`qtools deploy failed: ${error.message}`);
        //logger.debug(error.stack || error.toString());
        throw error;
    }
}
