import React, { useEffect, useMemo, useRef } from 'react';
import PropTypes from 'prop-types';
import classNames from 'classnames';
import Markdown from 'markdown-to-jsx';
import { Checkbox, Divider, Form, Input, Popover, Radio, Select, Typography } from 'antd';
import { ExclamationCircleOutlined } from '@ant-design/icons';
import { DOCUMENT_TYPES, VERDICT_OK, VERDICT_OTHER } from '../../domain/documents';
import { translate } from '../../utils/translate';
import { withHotkey } from '../Utility/WithHotkey';
import { Criteria } from './Criteria';
import { PassportBio } from './PassportBio';
import { PassportReg } from './PassportReg';
import { LicenseFront } from './LicenseFront';
import { LicenseBack } from './LicenseBack';
import './Document.less';

export const formLayout = {
    labelCol: { span: 8 },
    wrapperCol: { offset: 1, span: 15 },
};

const multipleSelectDocuments = ['face_compare', 'passport_selfie'];

export const DocumentView = ({
    documentType,
    documentTitle,
    fieldsData = {},
    feedback,
    facts,
    scaleOptions,
    verdicts,
    onVerdictChange,
    onAdditionalFieldsChange,
    onRegisterForms,
    taskError,
    resetTaskError,
    activeDocumentIndex,
    documentIndex,
    onAddressSearch,
    secret,
}) => {
    const verdictsFormName = `verdicts_${documentType}`;
    const additionalFormName = `additional_${documentType}`;

    const [verdictsForm] = Form.useForm();
    const [additionalForm] = Form.useForm();
    const [documentForm] = Form.useForm();
    const commentRef = useRef(null);
    const sectionRef = useRef(null);

    const hotkeyEventsDisabled = documentIndex !== activeDocumentIndex;

    const multipleSelect = scaleOptions.multipleSelect || multipleSelectDocuments.includes(documentType);

    const initialValues = useMemo(
        () =>
            Object.entries(fieldsData).reduce(
                (values, [fieldName, fieldValue]) => ({
                    ...values,
                    [fieldName]: fieldValue,
                }),
                {},
            ),
        [],
    );

    const fields = Object.entries(facts).map(([name, label]) => ({
        name,
        label,
    }));

    const availableFields = Object.keys(initialValues);

    const criteria = fields.map((field, index) => (
        <Criteria
            key={field.name}
            name={field.name}
            label={field.label}
            index={index}
            multipleSelect={multipleSelect}
            disabled={hotkeyEventsDisabled}
        />
    ));

    const dependencies = [...fields.map(({ name }) => [name]), [VERDICT_OTHER]];

    const InputComponent = multipleSelect ? withHotkey(Checkbox) : withHotkey(Radio);

    useEffect(() => onRegisterForms(documentType, [verdictsForm, additionalForm]), []);

    const handleVerdictChange = (values) => {
        if (!values[0]) {
            return;
        }

        resetTaskError();

        const { name, value } = values[0];
        const namePath = name.join('.');

        if (namePath === VERDICT_OK && value) {
            verdictsForm.setFieldsValue({
                ...fields.reduce((results, { name: field }) => ({ ...results, [field]: false }), {}),
                [VERDICT_OTHER]: false,
            });

            return onVerdictChange({ verdicts: { [documentType]: [VERDICT_OK] }, comments: { [document]: null } });
        }

        if (fields.some(({ name }) => name === namePath) || namePath === VERDICT_OTHER) {
            verdictsForm.setFieldsValue({ [VERDICT_OK]: false });

            if (!multipleSelect) {
                verdictsForm.setFieldsValue({
                    ...fields.reduce((results, { name: field }) => ({ ...results, [field]: false }), {}),
                    [VERDICT_OTHER]: false,
                    [namePath]: value,
                });
            }

            let results = verdicts.filter((verdict) => verdict !== VERDICT_OK && verdict !== namePath);

            results = value ? (multipleSelect ? [...results, namePath] : [namePath]) : results;

            return onVerdictChange({ verdicts: { [documentType]: results } });
        }

        if (namePath === 'description') {
            onVerdictChange({ comments: { [documentType]: value } });
        }
    };

    const handleAdditionalFieldsChange = (values) => {
        if (!values[0]) {
            return;
        }

        const { name, value } = values[0];

        onAdditionalFieldsChange({ name, value });
    };

    const handleCommentKeyDown = (e) => {
        const hasModifier = e.ctrlKey || e.metaKey;

        if (e.key !== 'Enter') {
            return;
        }

        if (!hasModifier) {
            return e.preventDefault();
        }

        commentRef.current?.blur();
    };

    const renderVerdictsForm = () => (
        <Form
            name={verdictsFormName}
            form={verdictsForm}
            onFieldsChange={handleVerdictChange}
            labelAlign="left"
            hideRequiredMark
            colon={false}
        >
            {criteria}

            {scaleOptions.otherWithComment && (
                <>
                    <Form.Item className="ant-form-item_compact" name={VERDICT_OTHER} valuePropName="checked">
                        <InputComponent hotKey={{ code: 'KeyQ', label: 'Q' }} disabled={hotkeyEventsDisabled}>
                            {translate('verdict_other')}
                        </InputComponent>
                    </Form.Item>

                    <Form.Item
                        noStyle
                        shouldUpdate={(prevValues, currentValues) =>
                            prevValues[VERDICT_OTHER] !== currentValues[VERDICT_OTHER]
                        }
                    >
                        {({ getFieldValue }) =>
                            getFieldValue([VERDICT_OTHER]) && (
                                <Form.Item
                                    name="description"
                                    rules={[
                                        ({ getFieldValue }) => ({
                                            validator(_, value) {
                                                const isRequired = getFieldValue([VERDICT_OTHER]);

                                                if (!value && isRequired) {
                                                    return Promise.reject(translate('error_validation_description'));
                                                }

                                                return Promise.resolve();
                                            },
                                        }),
                                    ]}
                                    preserve={false}
                                >
                                    <Input
                                        ref={commentRef}
                                        autoFocus
                                        className="sidebar__description"
                                        placeholder={translate('verdict_description_placeholder')}
                                        onKeyDown={handleCommentKeyDown}
                                        disabled={hotkeyEventsDisabled}
                                    />
                                </Form.Item>
                            )
                        }
                    </Form.Item>
                </>
            )}

            <Form.Item
                name={VERDICT_OK}
                dependencies={dependencies}
                valuePropName="checked"
                validateStatus={taskError ? 'error' : null}
                help={taskError}
                rules={[
                    ({ getFieldsValue }) => ({
                        validator(_, value) {
                            const formValues = getFieldsValue();

                            if (!value && Object.keys(formValues).every((field) => !formValues[field])) {
                                return Promise.reject(translate('error_validation_criteria'));
                            }
                            return Promise.resolve();
                        },
                    }),
                ]}
            >
                <InputComponent hotKey={{ code: 'KeyW', label: 'W' }} disabled={hotkeyEventsDisabled}>
                    {translate('verdict_ok')}
                </InputComponent>
            </Form.Item>
        </Form>
    );

    const renderAdditionalFieldItem = ([name, { label, required, values }]) => (
        <Form.Item
            key={name}
            label={label}
            name={name}
            rules={[{ required, message: translate('error_field_required') }]}
            preserve={false}
        >
            <Select allowClear>
                {Object.entries(values).map(([option, label]) => (
                    <Select.Option key={option} value={option}>
                        {label}
                    </Select.Option>
                ))}
            </Select>
        </Form.Item>
    );

    const renderAdditionalFields = () => (
        <>
            <Divider />
            <Form
                name={additionalFormName}
                form={additionalForm}
                onFieldsChange={handleAdditionalFieldsChange}
                labelAlign="left"
                hideRequiredMark
                colon={false}
                {...formLayout}
            >
                {Object.entries(scaleOptions.additionalFields).map(renderAdditionalFieldItem)}
            </Form>
        </>
    );

    const packJson = require('../../../package.json');
    const FeedBackContent = () => (
        <>
            <Markdown>{feedback}</Markdown>

            <section className="popover__secret">
                <div>Secret: {secret}</div>
                <div>Version: {packJson?.version ?? '-'}</div>
            </section>
        </>
    );

    const renderFeedBack = () => (
        <Popover
            defaultVisible
            destroyTooltipOnHide
            trigger="click"
            position="bottomLeft"
            overlayClassName="document__feedback"
            content={FeedBackContent}
            title={translate('document_feedback')}
            getPopupContainer={() => sectionRef.current}
        >
            <ExclamationCircleOutlined className="document__feedback-icon" />
        </Popover>
    );

    const title = scaleOptions.specialHeaders?.[documentType] ?? documentTitle;

    return (
        <section className={classNames('document', documentType)} ref={sectionRef}>
            <Typography.Title level={4} type="secondary" className="document__header">
                {title}
                {feedback && renderFeedBack()}
            </Typography.Title>

            {Object.keys(fieldsData).length ? (
                <>
                    <section className="document__fields">
                        <Form
                            name={documentType}
                            form={documentForm}
                            labelAlign="left"
                            initialValues={initialValues}
                            colon={false}
                            {...formLayout}
                        >
                            {documentType === DOCUMENT_TYPES.passportBio && (
                                <PassportBio availableFields={availableFields} />
                            )}
                            {documentType === DOCUMENT_TYPES.passportReg && (
                                <PassportReg availableFields={availableFields}
                                    onAddressSearch={onAddressSearch} />
                            )}
                            {documentType === DOCUMENT_TYPES.licenseFront && (
                                <LicenseFront availableFields={availableFields} />
                            )}
                            {documentType === DOCUMENT_TYPES.licenseBack && (
                                <LicenseBack availableFields={availableFields} />
                            )}
                        </Form>
                    </section>

                    <Divider />
                </>
            ) : null}

            {renderVerdictsForm()}

            {scaleOptions.additionalFields && renderAdditionalFields()}
        </section>
    );
};

DocumentView.propTypes = {
    documentType: PropTypes.string.isRequired,
    documentTitle: PropTypes.string.isRequired,
    fieldsData: PropTypes.shape({ [PropTypes.string.isRequired]: PropTypes.any }),
    feedback: PropTypes.string,
    facts: PropTypes.object.isRequired,
    verdicts: PropTypes.arrayOf(PropTypes.string),
    onVerdictChange: PropTypes.func.isRequired,
    onAdditionalFieldsChange: PropTypes.func.isRequired,
    onRegisterForms: PropTypes.func.isRequired,
    scaleOptions: PropTypes.shape({
        multipleSelect: PropTypes.bool,
        otherWithComment: PropTypes.bool,
        additionalFields: PropTypes.object,
        specialHeaders: PropTypes.object,
    }),
    taskError: PropTypes.string,
    resetTaskError: PropTypes.func.isRequired,
    activeDocumentIndex: PropTypes.number,
    documentIndex: PropTypes.number,
    onAddressSearch: PropTypes.func,
    secret: PropTypes.string,
};
