import React, { useMemo } from 'react';
import PropTypes from 'prop-types';
import { Form, Select } from 'antd';
import { CalendarOutlined } from '@ant-design/icons';
import { i18next, translate } from '../../../utils/translate';
import { GENDERS } from '../../../domain/documents';
import { COUNTRIES, EXTRA_COUNTRIES } from '../../../domain/countries';
import { TextField } from '../TextField';

export const PassportBio = ({ availableFields, disabled = true }) => {
    const locale = i18next.language;
    const countries = useMemo(
        () =>
            [...EXTRA_COUNTRIES, ...COUNTRIES].map((country) => ({ value: country.altValue, label: country[locale] })),
        [],
    );

    const renderField = (fieldName, index) => {
        const tabIndex = index + 1;
        const optionalHint = ['birth_place', 'middle_name'].includes(fieldName)
            ? translate('field_optional_value_hint')
            : null;

        switch (fieldName) {
            case 'last_name':
            case 'first_name':
            case 'middle_name':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name={fieldName}
                        placeholder={translate(`field_${fieldName}_placeholder`)}
                        hint={optionalHint}
                        disabled={disabled}
                    />
                );

            case 'gender':
                return (
                    <Form.Item key={fieldName} label={translate('field_gender')} name="gender">
                        <Select
                            placeholder={translate('field_gender_placeholder')}
                            tabIndex={tabIndex}
                            allowClear
                            disabled={disabled}
                        >
                            {Object.values(GENDERS).map((gender) => (
                                <Select.Option key={gender} value={gender}>
                                    {translate(`gender_${gender}`)}
                                </Select.Option>
                            ))}
                        </Select>
                    </Form.Item>
                );

            case 'citizenship':
                return (
                    <Form.Item key={fieldName} label={translate('field_citizenship')} name="citizenship">
                        <Select
                            placeholder={translate('field_citizenship_placeholder')}
                            tabIndex={tabIndex}
                            options={countries}
                            showSearch
                            allowClear
                            optionFilterProp="label"
                            disabled={disabled}
                        />
                    </Form.Item>
                );

            case 'birth_place':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="birth_place"
                        hint={optionalHint}
                        disabled={disabled}
                    />
                );

            case 'birth_date':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="birth_date"
                        placeholder={translate('field_date_placeholder')}
                        addonBefore={<CalendarOutlined />}
                        mask="1111-11-11"
                        disabled={disabled}
                    />
                );

            case 'number':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="number"
                        placeholder={translate('field_passport_number_placeholder')}
                        disabled={disabled}
                    />
                );

            case 'issue_date':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="issue_date"
                        placeholder={translate('field_date_placeholder')}
                        addonBefore={<CalendarOutlined />}
                        mask="1111-11-11"
                        disabled={disabled}
                    />
                );
            case 'subdivision_code':
                return (
                    <TextField
                        key={fieldName}
                        tabIndex={tabIndex}
                        name="subdivision_code"
                        placeholder={translate('field_subdivision_code_placeholder')}
                        mask="111-111"
                        disabled={disabled}
                    />
                );
        }
    };

    return availableFields.map(renderField);
};

PassportBio.propTypes = {
    availableFields: PropTypes.arrayOf(PropTypes.string.isRequired).isRequired,
    disabled: PropTypes.bool,
};
