/* eslint-disable camelcase */
import React, { useLayoutEffect, useEffect, useState } from 'react';
import PropTypes from 'prop-types';
import { Tabs, Form } from 'antd';
import { serviceLocator } from '../../infrastructure/serviceLocator';
import UsecaseRunError from '../../infrastructure/errors/UsecaseRunError';
import { translate } from '../../utils/translate';
import { useHotkeys } from '../../hooks/useHotkeys/useHotkeys';
import { useFocus } from '../../hooks/useFocus/useFocus';
import { DOCUMENT_TYPES } from '../../domain/documents';
import { Hint } from '../Utility/Hint';
import { Document } from '../Document';
import './Sidebar.less';

export const SidebarView = ({ activeDocumentIndex, setActiveDocumentIndex, documentTypes, presets, secret }) => {
    const [sidebar, focusSidebar] = useFocus();
    const [formInstances, setFormInstances] = useState({});

    useLayoutEffect(() => focusSidebar(), []);

    useEffect(() => {
        serviceLocator.validateAndCollect = async () => {
            try {
                const instances = Object.values(formInstances).flat();

                await Promise.all(instances.map(async ({ validateFields }) => validateFields()));
            } catch {
                throw new UsecaseRunError();
            }
        };

        return () => {
            serviceLocator.validateAndCollect = null;
        };
    }, [formInstances]);

    const handleGoBack = () => {
        setActiveDocumentIndex(Math.max(0, activeDocumentIndex - 1));
    };

    const handleGoForward = () => {
        setActiveDocumentIndex(Math.min(documentTypes.length - 1, activeDocumentIndex + 1));
    };

    useHotkeys(
        [
            { keys: ['ArrowUp'], onEvent: () => handleGoBack() },
            { keys: ['ArrowDown'], onEvent: () => handleGoForward() },
            { keys: ['ArrowLeft'], onEvent: () => handleGoBack() },
            { keys: ['ArrowRight'], onEvent: () => handleGoForward() },
        ],
        [activeDocumentIndex],
    );

    const handleFormsRegistration = (documentId, instances) => {
        setFormInstances((currInstances) => ({ ...currInstances, [documentId]: instances }));
    };

    const handleTabChange = (activeKey) => setActiveDocumentIndex(Number(activeKey));

    const renderTabHeader = (title, index) => (
        <Hint title={title} placement="left">
            {index + 1}
        </Hint>
    );

    const renderTab = (documentId, index) => {
        const isPreset = !Object.values(DOCUMENT_TYPES).includes(documentId);
        const displayedTypes = isPreset ? presets[documentId].document_types : [documentId];
        const title = displayedTypes.map(translate).join(', ');

        return (
            <Tabs.TabPane tab={renderTabHeader(title, index)} key={String(index)} forceRender>
                <Document
                    documentType={documentId}
                    documentTitle={title}
                    documentIndex={index}
                    secret={secret}
                    onRegisterForms={handleFormsRegistration}
                />
            </Tabs.TabPane>
        );
    };

    return (
        <div className="sidebar" ref={sidebar} tabIndex={-1}>
            <Form.Provider>
                <Tabs
                    className="sidebar__tabs"
                    tabPosition="right"
                    activeKey={String(activeDocumentIndex)}
                    keyboard={true}
                    onChange={handleTabChange}
                >
                    {documentTypes.map(renderTab)}
                </Tabs>
            </Form.Provider>
        </div>
    );
};

SidebarView.propTypes = {
    activeDocumentIndex: PropTypes.number.isRequired,
    setActiveDocumentIndex: PropTypes.func.isRequired,
    documentTypes: PropTypes.arrayOf(PropTypes.string).isRequired,
    presets: PropTypes.shape({
        [PropTypes.string]: PropTypes.shape({
            document_types: PropTypes.arrayOf(PropTypes.oneOf(Object.values(DOCUMENT_TYPES))).isRequired,
            is_video_available: PropTypes.bool,
        }),
    }),
    secret: PropTypes.string,
    formErrors: PropTypes.object,
};
