/* eslint-disable camelcase */
import React, { Component } from 'react';
import PropTypes from 'prop-types';
import Storage from '../../utils/storage';
import { translate } from '../../utils/translate';
import SplitPane from 'react-split-pane';
import { Result } from 'antd';
import { DOCUMENT_TYPES } from '../../domain/documents';
import { ErrorIcon } from '../Icons/ErrorIcon';
import { ViewerPane } from '../ViewerPane';
import { Sidebar } from '../Sidebar';
import { Async } from '../Utility/Async';
import { Spin } from '../Spin';
import './TaskPageLayout.less';

// eslint-disable-next-line no-magic-numbers
const EXPIRATION_TIME = 365 * 24 * 60 * 60 * 1000;
const DEFAULT_PANE_SIZE = 510;

export class TaskPageLayoutView extends Component {
    static propTypes = {
        secret: PropTypes.string,
        documentTypes: PropTypes.arrayOf(PropTypes.string).isRequired,
        presets: PropTypes.shape({
            [PropTypes.string]: PropTypes.shape({
                document_types: PropTypes.arrayOf(PropTypes.oneOf(Object.values(DOCUMENT_TYPES))).isRequired,
                is_video_available: PropTypes.bool,
            }),
        }),
        loadDocuments: PropTypes.func.isRequired,
    };

    get splitSize() {
        return this.storage.getItem('driveVideoSplitSize') || DEFAULT_PANE_SIZE;
    }

    constructor(props) {
        super(props);

        this.storage = new Storage();
    }

    componentWillUnmount() {
        this.storage.destroy();
    }

    handleSplitSize = (size) =>
        this.storage.setItem('driveVideoSplitSize', size, new Date().getTime() + EXPIRATION_TIME);

    renderErrorLayout = (error) => (
        <Result
            className="task-pane__error"
            title={translate('error_init_task')}
            subTitle={error.message}
            icon={<ErrorIcon width="100%" height="100%" />}
        />
    );

    renderLoadingLayout = () => <Spin size={32} className="task-pane__loader" />;

    renderMainLayout = ({ documentTypes, presets, secret }) => (
        <div className="task-pane">
            <div className="task-pane__row">
                <SplitPane
                    primary="second"
                    split="vertical"
                    minSize={DEFAULT_PANE_SIZE}
                    defaultSize={this.splitSize}
                    onDragFinished={this.handleSplitSize}
                >
                    <main className="task-pane__main">
                        <ViewerPane documentTypes={documentTypes} presets={presets} />
                    </main>

                    <aside className="task-pane__aside">
                        <Sidebar documentTypes={documentTypes} presets={presets} secret={secret} />
                    </aside>
                </SplitPane>
            </div>
        </div>
    );

    render() {
        const { documentTypes, presets, secret, loadDocuments } = this.props;

        return (
            <Async
                promise={loadDocuments({ documentTypes, presets, secret })}
                renderErrorLayout={this.renderErrorLayout}
                renderLoadingLayout={this.renderLoadingLayout}
            >
                {() => this.renderMainLayout({ documentTypes, presets, secret })}
            </Async>
        );
    }
}
