import React, { useState, useMemo } from 'react';
import PropTypes from 'prop-types';
import classNames from 'classnames';
import { Button } from 'antd';
import { PictureOutlined, VideoCameraOutlined } from '@ant-design/icons';
import { DOCUMENT_TYPES } from '../../domain/documents';
import { translate } from '../../utils/translate';
import { VideoPlayer } from '../VideoPlayer';
import { PhotoViewer } from '../PhotoViewer';
import './ViewerPane.less';

export const ViewerPaneView = ({ documents, isVideoAvailable, isVideoPreferred, secret, getMediaLink }) => {
    const [isVideoShown, setVideoShown] = useState(isVideoAvailable && isVideoPreferred);
    const modeLabel = translate(isVideoShown ? 'common_mode_photo' : 'common_mode_video');
    const modeIcon = isVideoShown ? <PictureOutlined /> : <VideoCameraOutlined />;

    const imageData = useMemo(
        () =>
            documents.map((document) => ({
                document,
                src: getMediaLink({ secret, document, type: 'image' }),
                alt: translate(document),
            })),
        [documents],
    );

    const videoData = useMemo(
        () => documents.map((document) => ({ document, src: getMediaLink({ secret, document, type: 'video' }) })),
        [documents],
    );

    const handleModeChange = () => setVideoShown(!isVideoShown);

    const renderVideoViewer = ({ document, src }) => (
        <VideoPlayer key={document} src={src} className={classNames('video-viewer', document)} />
    );

    const renderPhotoViewer = ({ document, src, alt }) => <PhotoViewer key={document} src={src} alt={alt} />;

    return (
        <div className="viewer-pane">
            {isVideoAvailable && (
                <Button className="viewer-pane__actions" onClick={handleModeChange} icon={modeIcon} type="primary">
                    {modeLabel}
                </Button>
            )}

            {isVideoShown && isVideoAvailable ? videoData.map(renderVideoViewer) : imageData.map(renderPhotoViewer)}
        </div>
    );
};

ViewerPaneView.propTypes = {
    documents: PropTypes.arrayOf(PropTypes.oneOf(Object.values(DOCUMENT_TYPES))).isRequired,
    isVideoAvailable: PropTypes.bool,
    isVideoPreferred: PropTypes.bool,
    secret: PropTypes.string.isRequired,
    getMediaLink: PropTypes.func.isRequired,
};
