import { resetCache } from '../infrastructure/Cache';
import { serviceLocator, serviceLocatorFactory } from '../infrastructure/serviceLocator';
import { ProxyREST } from '../infrastructure/transport/ProxyREST';
import { i18nextInit } from '../utils/translate';

export default class DriveTaskSuite extends window.TolokaTaskSuite {
    constructor(options) {
        super(options);

        this.locale = ['ru', 'ua', 'by', 'kz', 'uz'].includes(this.getWorkspaceOptions().language) ? 'ru' : 'en';

        try {
            resetCache();
        } catch (e) {
            console.warn('Reset cache error!', e);
        }

        const baseTransportParams = {
            sandboxChannel: this.getAssignment().getSandboxChannel(),
            assignmentId: this.getAssignment().getId(),
            origin: this.getWorkspaceOptions().origin,
        };

        serviceLocatorFactory(
            {
                MediaREST: new ProxyREST({
                    ...baseTransportParams,
                    baseUrl: this.getProxyNames().media,
                }),
                DataREST: new ProxyREST({
                    ...baseTransportParams,
                    baseUrl: this.getProxyNames().data,
                }),
                GeoREST: new ProxyREST({
                    ...baseTransportParams,
                    baseUrl: this.getProxyNames().geo,
                })
            },
            { replace: true },
        );
    }

    render() {
        i18nextInit({
            lng: this.locale,
            userKeys: this.getUserLocale(),
        }).then(() => {
            super.render();
        });

        return this;
    }

    getDOMElement() {
        const taskSuiteElement = super.getDOMElement();

        taskSuiteElement.classList.add('drive');

        return taskSuiteElement;
    }

    /**
     * Get rid of hotkeys, since they mess up React modals
     */
    initHotkeys() {
        this.hotkey.destroy();
        this.getTasks().forEach((task) => task.hotkey.destroy());
        this.focusTask(0);
    }

    /**
     * Getter for user-specified locale
     * @returns {{en: {}, ru: {}}}
     */
    getUserLocale() {
        return {
            ru: {},
            en: {},
        };
    }

    /**
     * Getter for user proxy names
     * @returns {{data: string, media: string, geo: string}}
     */
    getProxyNames() {
        return {
            data: 'robot-yang-cars-registration-data',
            media: 'robot-yang-cars-registration',
            geo: 'robot-yang-cars-search-geo',
        };
    }

    getFacts() {
        return {};
    }

    getPresets() {
        return {};
    }

    /**
     * Getter for misc scale options
     * @returns {{multipleSelect: boolean, otherWithComment: boolean}}
     */
    getScaleOptions() {
        return {
            multipleSelect: false,
            otherWithComment: false,
        };
    }

    /**
     * @param {array} solutions
     * @returns {Promise<*>}
     */
    async validate(solutions) {
        let isValid = true;
        const suiteErrors = [];

        for (const solution of solutions) {
            const taskErrors = await this.getTasksIndexed()[solution.task_id].validate(solution);

            suiteErrors.push(taskErrors);

            if (taskErrors) {
                isValid = false;
            }
        }

        if (isValid) {
            return null;
        } else {
            this.onValidationFail(suiteErrors);
            return suiteErrors;
        }
    }
}
